require File.dirname(__FILE__) + '/../spec_helper'

describe CelebritiesController, "#route_for" do

  it "should map { :controller => 'celebrities', :action => 'index' } to /celebrities" do
    route_for(:controller => "celebrities", :action => "index").should == "/celebrities"
  end
  
  it "should map { :controller => 'celebrities', :action => 'new' } to /celebrities/new" do
    route_for(:controller => "celebrities", :action => "new").should == "/celebrities/new"
  end
  
  it "should map { :controller => 'celebrities', :action => 'show', :id => 1 } to /celebrities/1" do
    route_for(:controller => "celebrities", :action => "show", :id => 1).should == "/celebrities/1"
  end
  
  it "should map { :controller => 'celebrities', :action => 'edit', :id => 1 } to /celebrities/1/edit" do
    route_for(:controller => "celebrities", :action => "edit", :id => 1).should == "/celebrities/1/edit"
  end
  
  it "should map { :controller => 'celebrities', :action => 'update', :id => 1} to /celebrities/1" do
    route_for(:controller => "celebrities", :action => "update", :id => 1).should == "/celebrities/1"
  end
  
  it "should map { :controller => 'celebrities', :action => 'destroy', :id => 1} to /celebrities/1" do
    route_for(:controller => "celebrities", :action => "destroy", :id => 1).should == "/celebrities/1"
  end
  
end

describe CelebritiesController, "handling GET /celebrities" do

  before do
    @celebrity = mock_model(Celebrity)
    Celebrity.stub!(:find).and_return([@celebrity])
  end
  
  def do_get
    get :index
  end
  
  it "should be successful" do
    do_get
    response.should be_success
  end

  it "should render index template" do
    do_get
    response.should render_template('index')
  end
  
  it "should find all celebrities" do
    Celebrity.should_receive(:find).with(:all).and_return([@celebrity])
    do_get
  end
  
  it "should assign the found celebrities for the view" do
    do_get
    assigns[:celebrities].should == [@celebrity]
  end
end

describe CelebritiesController, "handling GET /celebrities.xml" do

  before do
    @celebrity = mock_model(Celebrity, :to_xml => "XML")
    Celebrity.stub!(:find).and_return(@celebrity)
  end
  
  def do_get
    @request.env["HTTP_ACCEPT"] = "application/xml"
    get :index
  end
  
  it "should be successful" do
    do_get
    response.should be_success
  end

  it "should find all celebrities" do
    Celebrity.should_receive(:find).with(:all).and_return([@celebrity])
    do_get
  end
  
  it "should render the found celebrity as xml" do
    @celebrity.should_receive(:to_xml).and_return("XML")
    do_get
    response.body.should == "XML"
  end
end

describe CelebritiesController, "handling GET /celebrities/1" do

  before do
    @celebrity = mock_model(Celebrity)
    Celebrity.stub!(:find).and_return(@celebrity)
  end
  
  def do_get
    get :show, :id => "1"
  end

  it "should be successful" do
    do_get
    response.should be_success
  end
  
  it "should render show template" do
    do_get
    response.should render_template('show')
  end
  
  it "should find the celebrity requested" do
    Celebrity.should_receive(:find).with("1").and_return(@celebrity)
    do_get
  end
  
  it "should assign the found celebrity for the view" do
    do_get
    assigns[:celebrity].should equal(@celebrity)
  end
end

describe CelebritiesController, "handling GET /celebrities/1.xml" do

  before do
    @celebrity = mock_model(Celebrity, :to_xml => "XML")
    Celebrity.stub!(:find).and_return(@celebrity)
  end
  
  def do_get
    @request.env["HTTP_ACCEPT"] = "application/xml"
    get :show, :id => "1"
  end

  it "should be successful" do
    do_get
    response.should be_success
  end
  
  it "should find the celebrity requested" do
    Celebrity.should_receive(:find).with("1").and_return(@celebrity)
    do_get
  end
  
  it "should render the found celebrity as xml" do
    @celebrity.should_receive(:to_xml).and_return("XML")
    do_get
    response.body.should == "XML"
  end
end

describe CelebritiesController, "handling GET /celebrities/new" do

  before do
    @celebrity = mock_model(Celebrity)
    Celebrity.stub!(:new).and_return(@celebrity)
  end
  
  def do_get
    get :new
  end

  it "should be successful" do
    do_get
    response.should be_success
  end
  
  it "should render new template" do
    do_get
    response.should render_template('new')
  end
  
  it "should create an new celebrity" do
    Celebrity.should_receive(:new).and_return(@celebrity)
    do_get
  end
  
  it "should not save the new celebrity" do
    @celebrity.should_not_receive(:save)
    do_get
  end
  
  it "should assign the new celebrity for the view" do
    do_get
    assigns[:celebrity].should equal(@celebrity)
  end
end

describe CelebritiesController, "handling GET /celebrities/1/edit" do

  before do
    @celebrity = mock_model(Celebrity)
    Celebrity.stub!(:find).and_return(@celebrity)
  end
  
  def do_get
    get :edit, :id => "1"
  end

  it "should be successful" do
    do_get
    response.should be_success
  end
  
  it "should render edit template" do
    do_get
    response.should render_template('edit')
  end
  
  it "should find the celebrity requested" do
    Celebrity.should_receive(:find).and_return(@celebrity)
    do_get
  end
  
  it "should assign the found celebrity for the view" do
    do_get
    assigns[:celebrity].should equal(@celebrity)
  end
end

describe CelebritiesController, "handling POST /celebrities" do

  before do
    @celebrity = mock_model(Celebrity, :to_param => "1")
    Celebrity.stub!(:new).and_return(@celebrity)
  end
  
  def post_with_successful_save
    @celebrity.should_receive(:save).and_return(true)
    post :create, :celebrity => {}
  end
  
  def post_with_failed_save
    @celebrity.should_receive(:save).and_return(false)
    post :create, :celebrity => {}
  end
  
  it "should create a new celebrity" do
    Celebrity.should_receive(:new).with({}).and_return(@celebrity)
    post_with_successful_save
  end

  it "should redirect to the new celebrity on successful save" do
    post_with_successful_save
    response.should redirect_to(celebrity_url("1"))
  end

  it "should re-render 'new' on failed save" do
    post_with_failed_save
    response.should render_template('new')
  end
end

describe CelebritiesController, "handling PUT /celebrities/1" do

  before do
    @celebrity = mock_model(Celebrity, :to_param => "1")
    Celebrity.stub!(:find).and_return(@celebrity)
  end
  
  def put_with_successful_update
    @celebrity.should_receive(:update_attributes).and_return(true)
    put :update, :id => "1"
  end
  
  def put_with_failed_update
    @celebrity.should_receive(:update_attributes).and_return(false)
    put :update, :id => "1"
  end
  
  it "should find the celebrity requested" do
    Celebrity.should_receive(:find).with("1").and_return(@celebrity)
    put_with_successful_update
  end

  it "should update the found celebrity" do
    put_with_successful_update
    assigns(:celebrity).should equal(@celebrity)
  end

  it "should assign the found celebrity for the view" do
    put_with_successful_update
    assigns(:celebrity).should equal(@celebrity)
  end

  it "should redirect to the celebrity on successful update" do
    put_with_successful_update
    response.should redirect_to(celebrity_url("1"))
  end

  it "should re-render 'edit' on failed update" do
    put_with_failed_update
    response.should render_template('edit')
  end
end

describe CelebritiesController, "handling DELETE /celebrity/1" do

  before do
    @celebrity = mock_model(Celebrity, :destroy => true)
    Celebrity.stub!(:find).and_return(@celebrity)
  end
  
  def do_delete
    delete :destroy, :id => "1"
  end

  it "should find the celebrity requested" do
    Celebrity.should_receive(:find).with("1").and_return(@celebrity)
    do_delete
  end
  
  it "should call destroy on the found celebrity" do
    @celebrity.should_receive(:destroy)
    do_delete
  end
  
  it "should redirect to the celebrities list" do
    do_delete
    response.should redirect_to(celebrities_url)
  end
end
