﻿using System;
using System.Threading.Tasks;
using System.Collections.Concurrent;

namespace Pipeline {

    ////////////////////////////////////////////////////////////////////////////////
    //
    // SimplePipeline implementiert einfache lineare Pipeline.
    // Ein- und Ausgabetyp aller Stufen ist gleich.
    // SimplePipeline behält die Reihenfolge der Eingaben nicht bei!
    //
    class SimplePipeline<TChunk> {
        private ConcurrentBag<Task> rootTasks;      // sammelt alle Tasks, die von hier gestartet werden;
                                                    // nötig für Wait(); Subtasks müssen mit 
                                                    // AttachedToParent erzeugt werden.

        private Stage<TChunk> rootStage;            // Start-Stufe
        private ConcurrentBag<TChunk> results;    // Ergebnis-Sammler (muss threadsicher sein)

        private Stage<TChunk> _last;                // letzte hinzugefuegte Stufe

        public SimplePipeline(Stage<TChunk> root) {
            rootStage = root;
            _last = rootStage;
            rootTasks = new ConcurrentBag<Task>();
            results = new ConcurrentBag<TChunk>();
        }

        public void Process(TChunk chunk) {
            Task t;
            t = Task.Factory.StartNew(
                () => rootStage.Process(chunk, results)
            );
            rootTasks.Add(t);
        }

        public void AddStage(Stage<TChunk> stage) {
            _last.AddNext(stage);
            _last = stage;
        }

        public void Wait() {
            Task.WaitAll(rootTasks.ToArray());
        }

        public ConcurrentBag<TChunk> GetResults() {
            return results;
        }

    }

    ////////////////////////////////////////////////////////////////////////////////
    //
    // Pipeline-Stufe von SimplePipeline.
    // Jede Stufe erzeugt eine Task für die nachfolgende Stufe.
    //
    class Stage<TChunk> {

        private Func<TChunk, TChunk> stageFunc;     // Funktion, die in der Stufe ausgeführt wird
        private Stage<TChunk> nextStage;            // nachfolgende Stufe

        public Stage(Func<TChunk, TChunk> func) {
            stageFunc = func;
        }

        public void AddNext(Stage<TChunk> next) {
            nextStage = next;
        }

        public void Process(TChunk chunk, ConcurrentBag<TChunk> results) {
            TChunk result = stageFunc(chunk);
            if (nextStage != null) {
                nextStage.Process(result, results);
            } else {
                // letze Stufe erzeugt ein Ergebnis
                results.Add(result);
            }
        }
    }
}
