/* ==========================================================================
 *
 * (c) 2010 Christoph Leisegang
 *
 * ========================================================================== */
package model.neo4j;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.neo4j.graphdb.Direction;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.ReturnableEvaluator;
import org.neo4j.graphdb.StopEvaluator;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.TraversalPosition;
import org.neo4j.graphdb.Traverser;

class BookmarkImpl extends BaseImpl implements Bookmark {

	private static final String KEY_TITLE = "title";
	private static final String KEY_URI = "uri";
	private static final String KEY_DESCRIPTION = "description";

	public BookmarkImpl(final Node theNode, final Neo4jDatabaseService dbService) {
		super(theNode, dbService);
	}

	private Iterator<Label> getLabelsIter(final User user) {
		return new Iterator<Label>() {
			private final Iterator<Node> iterator = getTheNode()
					.traverse(
							Traverser.Order.BREADTH_FIRST,
							StopEvaluator.DEPTH_ONE,
							new ReturnableEvaluator() {
								public boolean isReturnableNode(
										final TraversalPosition pos) {
									final Node label = pos.currentNode();

									final Traverser traverser = label.traverse(
											Traverser.Order.BREADTH_FIRST,
											StopEvaluator.DEPTH_ONE,
											new ReturnableEvaluator() {
												public boolean isReturnableNode(
														final TraversalPosition pos) {
													return pos
															.currentNode()
															.equals(
																	((UserImpl) user)
																			.getTheNode());
												}
											},
											ServiceRelationshipTypes.USER_TO_LABEL,
											Direction.INCOMING);

									return !pos.isStartNode()
											&& traverser.iterator().hasNext();
								}
							}, ServiceRelationshipTypes.BM_TO_LABEL,
							Direction.OUTGOING).iterator();

			public boolean hasNext() {
				return iterator.hasNext();
			}

			public Label next() {
				final Node nextNode = iterator.next();
				return new LabelImpl(nextNode, getDatabaseService());
			}

			public void remove() {
				iterator.remove();
			}
		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see model.neo4j.Bookmark#getLabels(model.User)
	 */
	public Collection<Label> getLabels(final User user) {

		final Collection<Label> labels = new ArrayList<Label>();
		final Iterator<Label> iter = getLabelsIter(user);

		final Transaction tx = getDatabaseService().beginTx();

		try {
			while (iter.hasNext()) {
				labels.add(iter.next());
			}

			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}

		return labels;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see model.neo4j.Bookmark#addLabels(java.util.Collection)
	 */
	@Override
	public void add(final Label label) {

		if (!isConnected(label)) {
			final Transaction tx = getDatabaseService().beginTx();

			try {
				getTheNode().createRelationshipTo(
						((LabelImpl) label).getTheNode(),
						ServiceRelationshipTypes.BM_TO_LABEL);

				tx.success();
			} catch (Exception e) {
				tx.failure();
			} finally {
				tx.finish();
			}
		}
	}

	private boolean isConnected(Label label) {

		boolean isConnected = false;
		Iterable<Relationship> relIter = getTheNode().getRelationships(
				ServiceRelationshipTypes.BM_TO_LABEL);
		for (Relationship relationship : relIter) {
			if (relationship.getEndNode().equals(label)) {
				isConnected = true;
				break;
			}
		}

		return isConnected;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see model.neo4j.Bookmark#getTitle()
	 */
	@Override
	public String getTitle() {
		final Transaction tx = getDatabaseService().beginTx();
		String title;

		try {
			title = (String) getTheNode().getProperty(KEY_TITLE);

			tx.success();
		} catch (final Exception e) {
			title = "";
			tx.failure();
		} finally {
			tx.finish();
		}

		return title;
	}

	public void setTitle(final String title) {

		final Transaction tx = getDatabaseService().beginTx();
		try {
			getTheNode().setProperty(KEY_TITLE, title);
			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see model.neo4j.Bookmark#getUri()
	 */
	@Override
	public String getUri() {
		String uri;
		final Transaction tx = getDatabaseService().beginTx();

		try {
			uri = (String) getTheNode().getProperty(KEY_URI);

			tx.success();
		} catch (final Exception e) {
			uri = "";
			tx.failure();
		} finally {
			tx.finish();
		}

		return uri;
	}

	public void setUri(final String uri) {

		final Transaction tx = getDatabaseService().beginTx();
		try {
			getTheNode().setProperty(KEY_URI, uri);
			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}
	}

	@Override
	public String getDescription() {

		final Transaction tx = getDatabaseService().beginTx();
		String description;

		try {
			description = (String) getTheNode().getProperty(KEY_DESCRIPTION);

			tx.success();
		} catch (final Exception e) {
			description = "";
			tx.failure();
		} finally {
			tx.finish();
		}

		return description;
	}

	public void setDescription(final String description) {
		final Transaction tx = getDatabaseService().beginTx();
		try {
			getTheNode().setProperty(KEY_DESCRIPTION, description);
			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}
	}

	@Override
	public void remove(Label label) {

		Transaction tx = getDatabaseService().beginTx();

		try {
			Iterable<Relationship> relIter = getTheNode().getRelationships(
					ServiceRelationshipTypes.BM_TO_LABEL);
			for (Relationship relationship : relIter) {
				if (relationship.getEndNode().equals(
						((LabelImpl) label).getTheNode())) {
					relationship.delete();
					break;
				}
			}

			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}
	}

	@Override
	public boolean equals(final Object obj) {
		if (obj instanceof BookmarkImpl) {
			return getTheNode().equals(((BookmarkImpl) obj).getTheNode());
		}
		return false;
	}

	@Override
	public int hashCode() {
		return getTheNode().hashCode();
	}

	@Override
	public String toString() {
		return String.format("BookmarkImpl [node id=%d, title=%s, uri=%s]",
				getTheNode().getId(), getTitle(), getUri());
	}
}
