// ----------------------------------------------------------------------------------
// Microsoft Developer & Platform Evangelism
// 
// Copyright (c) Microsoft Corporation. All rights reserved.
// 
// THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, 
// EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
// OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
// ----------------------------------------------------------------------------------
// The example companies, organizations, products, domain names,
// e-mail addresses, logos, people, places, and events depicted
// herein are fictitious.  No association with any real company,
// organization, product, domain name, email address, logo, person,
// places, or events is intended or should be inferred.
// ----------------------------------------------------------------------------------

namespace WrapTools
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Net;
    using System.ServiceModel;
    using System.ServiceModel.Web;

    public abstract class AbstractWrapAuthorizationManager : ServiceAuthorizationManager
    {
        private TokenValidator validator;

        public void Init(string issuerName, string trustedAudienceValue, byte[] trustedSigningKey)
        {
            this.validator = new TokenValidator(issuerName, trustedAudienceValue, trustedSigningKey);
        }

        protected override bool CheckAccessCore(OperationContext operationContext)
        {
            // get the authorization header
            string authorizationHeader = WebOperationContext.Current.IncomingRequest.Headers[HttpRequestHeader.Authorization];

            if (string.IsNullOrEmpty(authorizationHeader))
            {
                SetUnauthorizedResponse();
                return false;
            }

            // check that it starts with 'WRAP'
            if (!authorizationHeader.StartsWith("WRAP "))
            {
                SetUnauthorizedResponse();
                return false;
            }

            string[] nameValuePair = authorizationHeader.Substring("WRAP ".Length).Split(new char[] { '=' }, 2);

            if (nameValuePair.Length != 2 ||
                nameValuePair[0] != "access_token" ||
                !nameValuePair[1].StartsWith("\"") ||
                !nameValuePair[1].EndsWith("\""))
            {
                SetUnauthorizedResponse();
                return false;
            }

            // trim the leading and trailing double-quotes
            string token = nameValuePair[1].Substring(1, nameValuePair[1].Length - 2);

            // validate the token
            if (!this.validator.Validate(token))
            {
                SetUnauthorizedResponse();
                return false;
            }

            // getClaims
            Dictionary<string, string> claims = this.validator.GetNameValues(token);

            // check for the correct action claim value
            if (CheckAccess(claims))
            {
                return true;
            }
            else
            {
                SetUnauthorizedResponse();
                return false;
            }
        }

        protected abstract bool CheckAccess(Dictionary<string, string> claims);

        protected void SetUnauthorizedResponse()
        {
            WebOperationContext.Current.OutgoingResponse.StatusCode = HttpStatusCode.Unauthorized;
            WebOperationContext.Current.OutgoingRequest.Headers.Add("WWW-Authenticate", "WRAP");
        }
    }
}