﻿using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Data.Entity.Validation;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace ITVisions
{
public class ValUtil
 {

  /// <summary>
  /// Validierung eines beliebigen annotierten .NET-Objekts
  /// </summary>
  public static List<ValidationResult> Validate(object obj)
  {
   var validationContext = new ValidationContext(obj);
   var validationResults = new List<ValidationResult>();
   // Validator.ValidateObject(obj, validationContext); führt bei Fehler zu Exception!
   Validator.TryValidateObject(obj, validationContext, validationResults, true);
   return validationResults;
  }


  /// <summary>
  /// Liefert Liste von Validierungsfehlern für normales .NET-Objekt
  /// </summary>
  public static Dictionary<string, List<string>> GetValResultAsKeyValuePair(List<ValidationResult> veList)
  {
   Dictionary<string, List<string>> e = new Dictionary<string, List<string>>();
   foreach (var ve in veList)
   {
    foreach (var prop in ve.MemberNames)
    {
     if (e.ContainsKey(prop))
     {
      e[prop].Add(ve.ErrorMessage);
     }
     else
     {
      e.Add(prop, new List<string> { ve.ErrorMessage });
     }
    }
   }
   return e;
  }

  /// <summary>
  /// Liefert Liste von Validierungsfehlern für EF-Objekt
  /// </summary>
  public static Dictionary<string, List<string>> GetValResultAsKeyValuePair(DbEntityValidationResult evr)
  {
   Dictionary<string, List<string>> e = new Dictionary<string, List<string>>();
   foreach (System.Data.Entity.Validation.DbValidationError ve in evr.ValidationErrors)
   {
    if (e.ContainsKey(ve.PropertyName))
    {
     e[ve.PropertyName].Add(ve.ErrorMessage);
    }
    else
    {
     e.Add(ve.PropertyName, new List<string> { ve.ErrorMessage });
    }
   }
   return e;
  }



  /// <summary>
  /// Ausgabe einer Liste von Validierungsfehlern für normales .NET-Objekt
  /// </summary>
  public static void PrintValErrors(List<ValidationResult> veList)
  {
   Console.ForegroundColor = ConsoleColor.Red;
   foreach (var ve in veList)
   {
    Console.WriteLine("########### Fehler: " + ve.ErrorMessage);
    foreach (var m in ve.MemberNames)
    {
     Console.WriteLine(" bei Eigenschaft " + m);
    }
   }
   Console.ForegroundColor = ConsoleColor.Gray;
  }

  /// <summary>
  /// Ausgabe einer Liste von Validierungsfehlern für EF_Objekt
  /// </summary>
  public static void PrintValErrors(IEnumerable<DbEntityValidationResult> evrList)
  {
   Console.ForegroundColor = ConsoleColor.Red;
   Console.WriteLine(GetValErrorsAsString(evrList));
   Console.ForegroundColor = ConsoleColor.Gray;
  }


  public static string GetValErrorsAsString(IDictionary<string, List<string>> liste)
  {
   int count = 0;
   string e = "";
   foreach (var eintrag in liste)
   {
    count++;
    e += ("########### Fehler " + count + " bei Eigenschaft " + eintrag.Key +"\n");
    foreach (string err in eintrag.Value)
    {
     e += err +"\n";
    }

   }
   return e;
  }

  public static string GetValErrorsAsString(IEnumerable<DbEntityValidationResult> fehlerListe) 
  {
   int count = 0;
   string ausgabe = "";
   foreach (var efObj in fehlerListe)
   {
    count++;
    ausgabe += (count + ". fehlerhaftes EF-Objekt " + efObj.Entry.Entity + "\n");
    foreach (System.Data.Entity.Validation.DbValidationError eintrag in efObj.ValidationErrors)
    {
    ausgabe+=("- " + eintrag.PropertyName + ": " + eintrag.ErrorMessage  + "\n");
    }
   }
   return ausgabe;
  }
 }
}
