package ix;

import ix.datamodel.QTeilnehmer;
import ix.datamodel.QVeranstaltung;
import ix.datamodel.Teilnehmer;
import ix.datamodel.Veranstaltung;

import java.text.DateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Random;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.PersistenceException;
import javax.servlet.annotation.WebServlet;

import org.datanucleus.api.jpa.JPAEntityManagerFactory;
import org.datanucleus.metadata.PersistenceUnitMetaData;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.mysema.query.jpa.impl.JPAQuery;
import com.vaadin.annotations.Theme;
import com.vaadin.annotations.VaadinServletConfiguration;
import com.vaadin.data.fieldgroup.BeanFieldGroup;
import com.vaadin.data.fieldgroup.FieldGroup;
import com.vaadin.data.util.BeanItem;
import com.vaadin.data.util.BeanItemContainer;
import com.vaadin.data.util.converter.StringToBooleanConverter;
import com.vaadin.data.util.converter.StringToDateConverter;
import com.vaadin.data.util.converter.StringToIntegerConverter;
import com.vaadin.data.validator.EmailValidator;
import com.vaadin.data.validator.IntegerRangeValidator;
import com.vaadin.data.validator.RegexpValidator;
import com.vaadin.server.Page;
import com.vaadin.server.VaadinRequest;
import com.vaadin.server.VaadinServlet;
import com.vaadin.ui.Button;
import com.vaadin.ui.CheckBox;
import com.vaadin.ui.ComboBox;
import com.vaadin.ui.DateField;
import com.vaadin.ui.FormLayout;
import com.vaadin.ui.Notification;
import com.vaadin.ui.Table;
import com.vaadin.ui.TextField;
import com.vaadin.ui.UI;
import com.vaadin.ui.VerticalLayout;
import com.vaadin.ui.Window;

@Theme("runo")
@SuppressWarnings("serial")
public class EventUI extends UI {

    @WebServlet(value = "/*", asyncSupported = true)
    @VaadinServletConfiguration(productionMode = false, ui = EventUI.class)
    public static class Servlet extends VaadinServlet {
    }

    private static EntityManagerFactory emf;

    // generate data sample
    static {
        generatePersistenceUnitAndInitializeEntityManagerFactory();
        generateVeranstaltung();
    }

    private final VerticalLayout rootLayout = new VerticalLayout();
    private final TeilnehmerForm teilnehmerForm = new TeilnehmerForm();

    private BeanItemContainer<Teilnehmer> teilnehmerContainer;
    private Table table;
    private Button teilnehmerHinzufügenButton;
    private ComboBox veranstaltungComboBox;
    private TextField filterVeranstaltung;
    private String filterTeilnehmer;
    private Window teilnehmerFormWindow;

    @Override
    protected void init(VaadinRequest request) {
        // Initialize page components
        initializeAddButton();
        initializeFilterTextField();
        initializeVeranstaltungComboBox();
        initializeContainerAndTable();
        initializeFormWindow();

        // Add components to root layout
        rootLayout.addComponent(teilnehmerHinzufügenButton);
        rootLayout.addComponent(veranstaltungComboBox);
        rootLayout.addComponent(filterVeranstaltung);
        rootLayout.addComponent(table);

        // Do some formatting
        rootLayout.setExpandRatio(table, 1.0f);
        rootLayout.setSizeFull();
        rootLayout.setMargin(true);
        rootLayout.setSpacing(true);

        // Set root layout as content
        setContent(rootLayout);

        // Refresh table
        refreshContainer();
    }

    /**
     * Initialize form window
     */
    private void initializeFormWindow() {
        teilnehmerFormWindow = new Window("Teilnehmerformular");
        teilnehmerFormWindow.setContent(teilnehmerForm);
        teilnehmerFormWindow.setModal(true);
        teilnehmerFormWindow.setResizable(false);
        teilnehmerFormWindow.center();
    }

    /**
     * Initialize add button
     */
    private void initializeAddButton() {
        teilnehmerHinzufügenButton = new Button("Neuen Teilnehmer hinzufügen");
        teilnehmerHinzufügenButton.addClickListener(e -> {
            teilnehmerForm.setItemDataSource(new Teilnehmer());
            addWindow(teilnehmerFormWindow);
        });
    }

    /**
     * Initialize container and table
     */
    @SuppressWarnings("unchecked")
    private void initializeContainerAndTable() {
        // Bean item container
        teilnehmerContainer = new BeanItemContainer<>(Teilnehmer.class);
        // Initialize new table, set container data source and configure columns
        table = new Table();
        table.setContainerDataSource(teilnehmerContainer);
        table.setVisibleColumns(QTeilnehmer.teilnehmer.name.getMetadata().getName(), //
                QTeilnehmer.teilnehmer.email.getMetadata().getName(),//
                QTeilnehmer.teilnehmer.geburtstag.getMetadata().getName(),//
                QTeilnehmer.teilnehmer.anzahlBegleitpersonen.getMetadata().getName(), //
                QTeilnehmer.teilnehmer.vortragender.getMetadata().getName());
        table.setColumnHeaders("Name", "E-Mail", "Geburtstag", "Anzahl Begleitpersonen", "Vortragender");

        // Configure boolean to string converter
        table.setConverter(QTeilnehmer.teilnehmer.vortragender.getMetadata().getName(), new StringToBooleanConverter() {

            @Override
            protected String getFalseString() {
                return "kein Vortragender";
            }

            @Override
            protected String getTrueString() {
                return "Vortragender";
            }
        });
     
        // Configure date to string converter
        table.setConverter(QTeilnehmer.teilnehmer.geburtstag.getMetadata().getName(), new StringToDateConverter() {

            @Override
            protected DateFormat getFormat(Locale locale) {
                DateFormat f = DateFormat.getDateInstance();
                f.setLenient(false);
                return f;
            }
        });

        // Make table selectable and handle item event
        table.setSelectable(true);
        table.addItemClickListener(e -> {
            BeanItem<Teilnehmer> teilnehmer = (BeanItem<Teilnehmer>) e.getItem();
            teilnehmerForm.setItemDataSource(teilnehmer.getBean());
            addWindow(teilnehmerFormWindow);
        });

        // formatting
        table.setSizeFull();
    }

    /**
     * Initialize event ComboBox filter
     */
    private void initializeVeranstaltungComboBox() {
        // Create ComboBox
        veranstaltungComboBox = createManyToOneVeranstaltungComboBox();
        // Handle value change event
        veranstaltungComboBox.addValueChangeListener(e -> refreshContainer());
    }
    
    private static ComboBox createManyToOneVeranstaltungComboBox() {
        // Bean item container
        BeanItemContainer<Veranstaltung> container = new BeanItemContainer<>(Veranstaltung.class);

        // Get an entity manager and select all events
        EntityManager em = emf.createEntityManager();
        try {
            container.addAll(new JPAQuery(em).from(QVeranstaltung.veranstaltung).list(QVeranstaltung.veranstaltung));
        } finally {
            em.close();
        }

        // Create ComboBox
    	ComboBox result = new ComboBox("Veranstaltung", container);
        // Configure entity property visible in comboBox
        result.setItemCaptionPropertyId(QVeranstaltung.veranstaltung.name.getMetadata().getName());
        return result;
    }

    /**
     * Refreshes the visitor table container.
     */
    private void refreshContainer() {
        // Remove all items from container
        teilnehmerContainer.removeAllItems();
        // Get an entity manager and select all visitors
        EntityManager em = emf.createEntityManager();
        try {
            JPAQuery query = new JPAQuery(em).from(QTeilnehmer.teilnehmer);
            List<Teilnehmer> list = executeTableQuery(query).list(QTeilnehmer.teilnehmer);
            teilnehmerContainer.addAll(list);
        } finally {
            em.close();
        }
    }

    /**
     * Configures a JPAQuery considering current filter values.
     * 
     * @param query
     *            JPAQuery to configure
     * @return configured JPAQuery
     */
    private JPAQuery executeTableQuery(JPAQuery query) {
        Veranstaltung veranstaltung = (Veranstaltung) veranstaltungComboBox.getValue();

        // If event filter is selected
        if (veranstaltung != null) {
            query = query.where(QTeilnehmer.teilnehmer.veranstaltung.eq(veranstaltung));
        }

        // If string filter is set
        if (!Strings.isNullOrEmpty(filterTeilnehmer)) {
            query = query.where(QTeilnehmer.teilnehmer.name.lower().contains(filterTeilnehmer.toLowerCase()). //
                    or(QTeilnehmer.teilnehmer.email.lower().contains(filterTeilnehmer.toLowerCase())));
        }
        return query;
    }

    /**
     * Initialize string filter as TextField
     */
    private void initializeFilterTextField() {
        filterVeranstaltung = new TextField("Filter");
        filterVeranstaltung.addTextChangeListener(e -> {
            filterTeilnehmer = e.getText();
            refreshContainer();
        });
    }

    /**
     * Generate 10 events with 20 visitors in each case on start up.
     */
    private static void generateVeranstaltung() {
        EntityManager em = emf.createEntityManager();
        em.getTransaction().begin();

        List<String> vornamen = Lists.newArrayList("Albert", "Peter", "Hans", "Jan");
        List<String> nachnahmen = Lists.newArrayList("Schneider", "Magnus", "Meller", "Alt");

        for (int i = 1; i < 10; i++) {
            Veranstaltung veranstaltung = new Veranstaltung();
            veranstaltung.setName("Expertenrunde " + i);
            veranstaltung.setDatum(new Date());

            Set<Teilnehmer> teilnehmerSet = new HashSet<>();
            for (int j = 0; j < 20; j++) {
                Teilnehmer teilnehmer = new Teilnehmer();
                String vorname = vornamen.get(new Random().nextInt(vornamen.size()));
                String nachnahme = nachnahmen.get(new Random().nextInt(nachnahmen.size()));
                teilnehmer.setName(String.format("%s %s", vorname, nachnahme));
                teilnehmer.setEmail(String.format("%s@%s.de", vorname.toLowerCase(), nachnahme.toLowerCase()));
                teilnehmer.setGeburtstag(new Date((long) (new Random().nextDouble() * 20 * 365 * 24 * 60 * 60 * 1000)));
                teilnehmer.setVeranstaltung(veranstaltung);
                teilnehmer.setAnzahlBegleitpersonen(new Random().nextInt(4));
                teilnehmer.setVortragender(new Random().nextBoolean());
                em.persist(teilnehmer);
                teilnehmerSet.add(teilnehmer);
            }
            veranstaltung.setTeilnehmer(teilnehmerSet);
            em.persist(veranstaltung);
        }
        em.getTransaction().commit();
        em.close();
    }

    /**
     * Create a in-memory demo database
     */
    private static void generatePersistenceUnitAndInitializeEntityManagerFactory() {
        PersistenceUnitMetaData pumd = new PersistenceUnitMetaData("dynamic-unit", "RESOURCE_LOCAL", null);
        pumd.addClassName(Teilnehmer.class.getCanonicalName());
        pumd.addClassName(Veranstaltung.class.getCanonicalName());
        pumd.setExcludeUnlistedClasses();
        pumd.addProperty("javax.persistence.jdbc.url", "jdbc:h2:mem:nucleus");
        pumd.addProperty("javax.persistence.jdbc.driver", "org.h2.Driver");
        pumd.addProperty("javax.persistence.jdbc.user", "sa");
        pumd.addProperty("javax.persistence.jdbc.password", "");
        pumd.addProperty("datanucleus.schema.autoCreateAll", "true");

        emf = new JPAEntityManagerFactory(pumd, null);
    }

    /**
     * Helper method saves an entity regarding its status (new or existing)
     * 
     * @param entity
     *            Entity to save
     */
    private static void save(Teilnehmer entity) {
        EntityManager em = emf.createEntityManager();
        try {
            em.getTransaction().begin();
            if (entity.getId() != null) {
                em.merge(entity);
            } else {
                em.persist(entity);
            }
            em.getTransaction().commit();
        } finally {
            em.close();
        }
    }

    /**
     * Visitor form
     */
    private class TeilnehmerForm extends FormLayout {

        private final TextField name = new TextField("Vor- und Nachname");
        private final TextField email = new TextField("E-Mailadresse");
        private final DateField geburtstag = new DateField("Geburtstag");
        private final TextField anzahlBegleitpersonen = new TextField("Anzahl Begleitpersonen");
        private final CheckBox vortragender = new CheckBox("trägt vor");
        private final ComboBox veranstaltung = createManyToOneVeranstaltungComboBox();
        private final Button saveButton = new Button();
        private final BeanFieldGroup<Teilnehmer> fieldGroup = new BeanFieldGroup<>(Teilnehmer.class);

        /**
         * Constructor initializes fields and components
         */
        TeilnehmerForm() {
            initFields();
            initButton();
            initFieldGroup();
            addComponents(name, email, geburtstag, veranstaltung, anzahlBegleitpersonen, vortragender, saveButton);
            setMargin(true);
        }

        /**
         * Initialize fields with validators and converters
         */
        private void initFields() {
            name.setNullRepresentation("");
            name.addValidator(new RegexpValidator(".+ .+", "Die Eingabe muss mindestens aus Vorname und Nachname bestehen"));
            email.setNullRepresentation("");
            email.addValidator(new EmailValidator("keine gültige E-Mail Adresse"));
            anzahlBegleitpersonen.setConverter(new StringToIntegerConverter());
            anzahlBegleitpersonen.addValidator(new IntegerRangeValidator("Sie dürfen max. 3 Personen mitbringen", 0, 3));
            veranstaltung.setRequired(true);
            veranstaltung.setNullSelectionAllowed(false);
        }

        /**
         * Initialize save button that commits the field group.
         */
        private void initButton() {
            saveButton.addClickListener(e -> {
                try {
                    fieldGroup.commit();
                } catch (FieldGroup.CommitException ex) {
                    Logger.getLogger(EventUI.class.getName()).log(Level.SEVERE, null, ex);
                }
            });
        }

        /**
         * Initialize the field group
         */
        private void initFieldGroup() {
            // Bind the form to the field group
            fieldGroup.bindMemberFields(this);
            // Add commit handler that will be executed an commit
            fieldGroup.addCommitHandler(new FieldGroup.CommitHandler() {

                @Override
                public void preCommit(FieldGroup.CommitEvent commitEvent) throws FieldGroup.CommitException {
                }

                /**
                 * After committing the form data successfully into the item:
                 * save the entity, close the window, give feedback and refresh
                 * the container
                 */
                @Override
                public void postCommit(FieldGroup.CommitEvent commitEvent) throws FieldGroup.CommitException {
                    Teilnehmer entity = fieldGroup.getItemDataSource().getBean();
                    try {
                    	save(entity);
                    } catch(PersistenceException e) {
                    	throw new FieldGroup.CommitException(e);
                    }
                    teilnehmerFormWindow.close();
                    Notification note = new Notification("Eintrag gespeichert.", Notification.Type.TRAY_NOTIFICATION);
                    note.show(Page.getCurrent());
                    refreshContainer();
                }
            });
        }

        /**
         * Load form with a new item
         * 
         * @param teilnehmer
         *            visitor item
         */
        public void setItemDataSource(Teilnehmer teilnehmer) {
            fieldGroup.setItemDataSource(teilnehmer);
            saveButton.setCaption(teilnehmer.getId() == null ? "anmelden" : "Änderungen speichern");
        }
    }
}
