package planner;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Laufzettel {
	// Maps tester (Integer) to station
	private HashMap<Integer, ArrayList<Integer>> testerStationHistory = new HashMap<Integer, ArrayList<Integer>>();
	
	// the names of the testers (names may be empty)
	private ArrayList<String> testers;

	// the names of the stations (names may be empty)
	private ArrayList<String> stations;

	public Laufzettel() {

		testers = new ArrayList<String>();
		stations = new ArrayList<String>();

	}

	public void setTesters(ArrayList<String> allTesters) {
		testers = allTesters;
		ArrayList<Integer> emptyArrayList = new ArrayList<Integer>();
		for (int roundIdx = 0; roundIdx < App.numRounds; roundIdx++) {
			emptyArrayList.add(-1);
		}

		for (int testerIdx = 0; testerIdx < testers.size(); testerIdx++) {
			// init the journey
			// it is vital to use clone here, otherwise we get a arrayList of references to the same element
			testerStationHistory.put(new Integer(testerIdx), (ArrayList<Integer>) emptyArrayList.clone());
		}

	}

	public void setStations(ArrayList<String> allStations) {
		stations = allStations;
	}

	/***
	 * Quickly visualize (i.e. make a string of) a single tester station history
	 * @param r
	 * @return
	 */
	private String join(ArrayList<Integer> r) {

		String res = "";
		for (Integer currentInt : r) {
			res += currentInt + ",";
		}
		if (res.length() > 0) {
			res = res.substring(0, res.length() - 1);
		}
		return "{" + res + "}";
	}


	/***
	 * Quickly visualize (i.e. make a string of) all tester's station history
	 * @param r
	 * @return
	 */
	public String ajoin() {
		String res = "[";
		for (Integer currentKey : testerStationHistory.keySet()) {
			res += currentKey + ":" + join(testerStationHistory.get(currentKey)) + ";\n";
		}
		return res + "]";
	}

	/**
	 * Read the template laufzettel-vorlage.html, replace the document placeholders like tester.name
	 * and the round-specific placeholders between <runden></runden> and write to laufzettel.html
	 */
	public void writeHTML() {

		String laufzettelVorlage = "";
		String templateFilename="laufzettel-vorlage.html";
		try (BufferedReader br = new BufferedReader(new FileReader(templateFilename))) {
			StringBuilder sb = new StringBuilder();
			String line = br.readLine();

			while (line != null) {
				sb.append(line);
				sb.append(System.lineSeparator());
				line = br.readLine();
			}
			laufzettelVorlage = sb.toString();
		}
		// Build the Solver
		catch (FileNotFoundException e) {
			String templateSampleContent="<!DOCTYPE HTML>\n"
					+ "<html lang='de'>"
					+ "<head>"
					+ ""
					+ "<title>Laufzettel</title>"
					+ "<meta charset='utf-8' />"
					+ "</head>"
					+ "<body>"
					+ "<br class='clear'/>"
					+ "<div class='content'>"
					+ "	<div class='bigger center'> Laufzettel <br class='clear'/><tester.name/></div>"
					+ "	<ul class='runden'>"
					+ "	<li class='clear'>"
					+ "	<div class='head runde'>Runde</div>"
					+ "	<!--div class='head station written'> Station</div-->"
					+ "	</li>"
					+ "		<runden>"
					+ "		<li class='clear'>"
					+ "			<div class='runde'><runde/>.</div>"
					+ "			<div class='station'> <station.name/></div>"
					+ "		</li>"
					+ "		</runden>"
					+ "	</ul>"
					+ "</div>"
					+ "</body>"
					+ "</html>";
			Path file = Paths.get(templateFilename);
			try {
				Files.write(file, templateSampleContent.getBytes());
				System.err.println("Beispiel-Laufzettel-Vorlagedatei "+templateFilename+" geschrieben.");
			} catch (IOException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}
			laufzettelVorlage=templateSampleContent;
		} catch (IOException e) {
			e.printStackTrace();
		}

		String laufzettelAlle = "";
		for (int testerIdx = 0; testerIdx < testers.size(); testerIdx++) {
			String laufzettelTester = new String(laufzettelVorlage);
			String pattern = "(?is)<runden>(.*?)<\\/runden>";

			String currentTestername = testers.get(testerIdx);
			if (currentTestername.length() == 0) {
				currentTestername = "Tester " + (testerIdx + 1);
			}
			// Create a Pattern object
			Pattern r = Pattern.compile(pattern);
			String outerGroup = "";
			String innerGroup = "";

			String appliedTemplate = "";

			// Now create matcher object.
			Matcher m = r.matcher(laufzettelTester);
			if (m.find()) {
				outerGroup = m.group(0);
				innerGroup = m.group(1);

			} else {
				System.err.println("Vorlage enthielt kein <runden>...</runden> Bereich");
			}

			for (int roundIdx = 0; roundIdx < App.numRounds; roundIdx++) {
				int station = get(testerIdx, roundIdx);
				String currentStationsname = "Pause";
				if (station >= 0) {
					currentStationsname = stations.get(station);
					if (currentStationsname.length() == 0) {
						currentStationsname = "Station " + (station + 1);
					}
				}
				appliedTemplate = appliedTemplate + innerGroup;
				appliedTemplate = appliedTemplate.replace("<tester.name/>", currentTestername);
				appliedTemplate = appliedTemplate.replace("<station.name/>", currentStationsname);
				appliedTemplate = appliedTemplate.replace("<runde/>", Integer.toString(roundIdx + 1));

			}
			laufzettelTester = laufzettelTester.replace(outerGroup, appliedTemplate);
			laufzettelTester = laufzettelTester.replace("<tester.name/>", currentTestername);
			laufzettelAlle = laufzettelAlle + laufzettelTester;

		}
		Path file = Paths.get("laufzettel.html");
		try {
			Files.write(file, laufzettelAlle.getBytes());
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

	}

	/***
	 * Writes the spreadsheets by_tester.csv and by_station.csv 
	 */
	public void writeCSVs() {
		StringBuilder csv = new StringBuilder();
		csv.append("Runde;Tester;Station;\n");
		int roundIdx = 0;
		for (roundIdx = 0; roundIdx < App.numRounds; roundIdx++) {
			for (int testerIdx = 0; testerIdx < testers.size(); testerIdx++) {

				int station = get(testerIdx, roundIdx);
				int roundNr = roundIdx + 1;
				int stationNr = station + 1;
				int testerNr = testerIdx + 1;
				csv.append(roundNr).append(";").append(testerNr).append(";").append(stationNr).append(";\n");

			}
		}

		Path file = Paths.get("by_tester.csv");
		try {
			Files.write(file, csv.toString().getBytes());
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		csv = new StringBuilder();
		csv.append("Runde;Station;Tester\n");

		for (roundIdx = 0; roundIdx < App.numRounds; roundIdx++) {
			for (int stationIdx = 0; stationIdx < stations.size(); stationIdx++) {
				ArrayList<Integer> stationTesters = getTesterByStationRound(stationIdx, roundIdx);
				int roundNr = roundIdx + 1;
				int stationNr = stationIdx + 1;
				for (Integer tester : stationTesters) {
					int testerNr = tester + 1;
					csv.append(roundNr).append(";").append(stationNr).append(";").append(testerNr).append(";\n");

				}

			}
		}
		file = Paths.get("by_station.csv");
		try {
			Files.write(file, csv.toString().getBytes());
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

	}

	/**
	 * Returns which tester(s) are assigned to a station in a particular round
	 * @param station
	 * @param runde
	 * @return
	 */
	public ArrayList<Integer> getTesterByStationRound(int station, int runde) {

		ArrayList<Integer> res = new ArrayList<>();

		for (int testerIdx = 0; testerIdx < testers.size(); testerIdx++) {
			if (testerStationHistory.get(testerIdx).get(runde) == station) {
				res.add(testerIdx);
			}
		}
		return res;

	}

	/**
	 * returns which station a tester is at at a particular round
	 * @param tester
	 * @param runde
	 * @return
	 */
	public int get(int tester, int runde) {
		return testerStationHistory.get(tester).get(runde);
	}

	/**
	 * places a tester at a particular station in a particular round
	 * @param tester
	 * @param runde
	 * @param station
	 */
	public void set(int tester, int runde, int station) {
		((ArrayList<Integer>) testerStationHistory.get(tester)).set(runde, station);
	}

	public boolean hasTesterVisitedStation(int tester, int station) {
		return (((ArrayList<Integer>) testerStationHistory.get(tester)).contains(station));
	}

	/***
	 * Fills a laufzettel from a testplan
	 * @param solvedTestplan
	 */
	public void fromTestPlan(Testplan solvedTestplan) {
		for (int roundIdx = 0; roundIdx < App.numRounds; roundIdx++) {
			for (Test process : solvedTestplan.getTestList()) {
				if (process.getRound() == roundIdx) {
					Station theStation = process.getStation();
					if (theStation != null) {
						set(process.getTester(), roundIdx, theStation.getIndex());
					}
				}
			}
		}
	}

}
