using ITVisions.Blazor;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Components.Authorization;
using Microsoft.AspNetCore.Hosting;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Hosting;
using Web.Data;
using Blazored.LocalStorage; // NuGet Blazor.Extensions.Storage
using MiracleList_Backend.Hubs;

namespace Web
{
 public class Startup
 {
  readonly Microsoft.AspNetCore.Hosting.IWebHostEnvironment Env;
  public IConfiguration Configuration { get; }

  public Startup(Microsoft.AspNetCore.Hosting.IWebHostEnvironment env)
  {
   this.Env = env;
   #region Load configuration
   // Get all configuration sources
   // NUGET: Microsoft.Extensions.Configuration.JSON
   // NUGET: Microsoft.Extensions.Configuration.EnvironmentVariables
   var builder = new ConfigurationBuilder()
       .SetBasePath(env.ContentRootPath)
       .AddInMemoryCollection()
       .AddJsonFile("appsettings.json", optional: true, reloadOnChange: true)
       .AddJsonFile($"appsettings.{env.EnvironmentName}.json", optional: true)
       .AddEnvironmentVariables(); // only loads process env variables https://github.com/aspnet/Configuration/issues/721

   // build configuration now
   Configuration = builder.Build();

   // get Connection String from configuration
   var CS = Configuration["ConnectionStrings:MiracleListDB"];

   // Inject connection string into DAL
   DAL.Context.IsRuntime = true;
   DAL.Context.ConnectionString = CS;
   #endregion
  }

  // This method gets called by the runtime. Use this method to add services to the container.
  // For more information on how to configure your application, visit https://go.microsoft.com/fwlink/?LinkID=398940
  public void ConfigureServices(IServiceCollection services)
  {
   // fr die Startseite _Host.cshtml
   services.AddRazorPages();
   // fr die Razor Components *.razor
   services.AddServerSideBlazor().AddCircuitOptions(o =>
   {
    if (this.Env.IsDevelopment()) //only add error details when debugging
    {
     o.DetailedErrors = true;
    }
   });

   #region Services fr MiracleList
   services.AddScoped<BlazorUtil>();
   services.AddScoped<AuthenticationStateProvider, MLAuthenticationStateProvider>();
   services.AddAuthorizationCore(); // sonst: System.InvalidOperationException: Cannot provide a value for property 'AuthorizationPolicyProvider' on type 'Microsoft.AspNetCore.Components.Authorization.AuthorizeView'. There is no registered service of type 'Microsoft.AspNetCore.Authorization.IAuthorizationPolicyProvider'.

   // NUGET: Blazored.LocalStorage;
   // GITHUB:  https://github.com/Blazored/LocalStorage --> 
   // Startup: using Blazored.LocalStorage / services.AddBlazoredLocalStorage();
   services.AddBlazoredLocalStorage();

   // Alternativ mglich (einige API-nderungen!)
   // NUGET: Blazor.Extensions.Storage
   // GITHUB: https://github.com/BlazorExtensions/Storage
   // using Blazor.Extensions.Storage;
   //services.AddBlazoredLocalStorage();
   #endregion

   #region Service fr andere Demos

   services.AddSingleton<WeatherForecastService>();

   //Fr Telerik
   //services.AddTelerikBlazor();

   //Fr Session state
   services.AddScoped<MLBlazorRCL.Komponentenzustand.TypedSessionState>();
   services.AddScoped<MLBlazorRCL.Komponentenzustand.GenericSessionState>();

   // fr Nutzung HTTPContext
   services.AddHttpContextAccessor();

   // fr HttpClient
   services.AddScoped<System.Net.Http.HttpClient>();
   #endregion

   #region SignalR
   services.AddSignalR();

   #endregion
  }

  // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
  public void Configure(IApplicationBuilder app, IWebHostEnvironment env)
  {
   if (env.IsDevelopment())
   {
    app.UseDeveloperExceptionPage();
   }
   else
   {
    app.UseExceptionHandler("/Home/Error");
    // The default HSTS value is 30 days. You may want to change this for production scenarios, see https://aka.ms/aspnetcore-hsts.
    app.UseHsts();
   }

   app.UseHttpsRedirection();
   app.UseStaticFiles();

   app.UseRouting();

   app.UseEndpoints(endpoints =>
   {
    endpoints.MapBlazorHub();
    endpoints.MapFallbackToPage("/_Host");
    // fr ASP.NET SignarlR
    endpoints.MapHub<MLHub>("/MLHub");
   });
  }
 }
}