package de.ix.lack;

import com.vaadin.flow.component.HasValue;
import com.vaadin.flow.component.formlayout.FormLayout;
import com.vaadin.flow.component.html.Div;
import com.vaadin.flow.component.textfield.TextField;
import com.vaadin.flow.data.binder.Binder;
import com.vaadin.flow.data.converter.StringToFloatConverter;
import com.vaadin.flow.data.validator.FloatRangeValidator;
import com.vaadin.flow.spring.annotation.UIScope;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.function.BiConsumer;
import java.util.function.Function;

@Component
@UIScope
public class DataModelComponent extends FormLayout {

    private final Binder<DataModel> binder = new Binder<>(DataModel.class);

    private final Raytrace rt = new Raytrace();
    private final LackService lackService;

    @Autowired
    public DataModelComponent(LackService lackService) {
        this.lackService = lackService;

        setResponsiveSteps(
                new ResponsiveStep("0", 3, ResponsiveStep.LabelsPosition.TOP),
                new ResponsiveStep("800px", 6, ResponsiveStep.LabelsPosition.ASIDE));

        Raytrace rt = new Raytrace();
        add(new Div(rt), 6);

        addColorControls("Farbe Umgebung", "ambientCol", DataModel::getAmbientCol, DataModel::setAmbientCol);
        addCoefficientControls("Intensität Umgebungslicht", "ambientCoefficient", DataModel::getAmbientCoefficient, DataModel::setAmbientCoefficient);

        addColorControls("Farbe Material", "diffuseCol", DataModel::getDiffuseCol, DataModel::setDiffuseCol);
        addCoefficientControls("Intensität Streulicht", "diffuseCoefficient", DataModel::getDiffuseCoefficient, DataModel::setDiffuseCoefficient);

        addColorControls("Farbe Glanzlicht", "specularCol", DataModel::getSpecularCol, DataModel::setSpecularCol);
        addCoefficientControls("Intensität Glanzlicht", "specularCoefficient", DataModel::getSpecularCoefficient, DataModel::setSpecularCoefficient);

        binder.addValueChangeListener(this::modelChanged);
    }

    private void addCoefficientControls(String label, String attributeName, Function<DataModel, Float> getCoefficient, BiConsumer<DataModel, Float> setCoefficient) {
        TextField textField = new TextField(label);
        binder.forField(textField).withConverter(
                new StringToFloatConverter("Bitte eine Zahl eingeben."))
                .withValidator(new FloatRangeValidator("Muss zwischen 0 und 1 liegen", 0f, 1f))
                .bind(attributeName);

        PaperSlider slider = new PaperSlider(100);
        binder.forField(slider).bind(
                dataModel -> (int) (getCoefficient.apply(dataModel) * 100),
                (dataModel, value)->setCoefficient.accept(dataModel, value/100.0f));

        add(textField, 3);
        add(slider, 3);
    }

    private void addColorControls(String label, String attributeName, Function<DataModel, float[]> getColor, BiConsumer<DataModel, float[]> setColor) {
        TextField textField = new TextField(label);
        binder.forField(textField).withConverter(
                new ColorConverter())
                .bind(attributeName);

        PaperSwatch paperSwatch = new PaperSwatch("#345678");
        binder.forField(paperSwatch).bind(
                dataModel -> ColorConverter.convertFloatColor(getColor.apply(dataModel)),
                (dataModel, value) -> setColor.accept(dataModel, ColorConverter.convertStringColor(value)));

        add(textField, 3);
        add(paperSwatch, 3);
    }

    private void modelChanged(HasValue.ValueChangeEvent<?> valueChangeEvent) {
        if (binder.isValid()) {
            String jsonModel = lackService.convertModelAsJson(binder.getBean());
            rt.repaint(jsonModel);

            // Aktualisierung der anderen Komponenten in der Benutzeroberfläche, triggert kein erneutes Repaint:
            binder.setBean(binder.getBean());
        }
    }

    public void setEntity(DataModel dataModel) {
        binder.setBean(dataModel);
        rt.repaint(lackService.convertModelAsJson(dataModel));
    }
}
