// Kleiner Raytracer um Materialien darzustellen.
// (c) Klaus Sausen, https://uc-mobileapps.com/de
// Der Schnitt einer Linie mit einer Kugel hat keine Lösung wenn der Strahl vorbei geht,
// eine Lösung wenn die Kugel tangiert wird und sonst zwei Lösungen beim Ein- und Austritt der Kugel.
let canvas;
let pixels;

Sphere = {
    setDataModel: function (canvasId, model) {
        canvas = document.getElementById (canvasId)
        this.context = canvas.getContext ("2d");
        this.image = this.context.getImageData (0, 0, canvas.width, canvas.height);
        pixels = this.image.data;

        this.material = new Material(model);
        this.updateCanvas ();
    },
    updateCanvas: function () {

        let eye = [0.0, 0.0, -4.0];
        let sphereMid = [0.0, 0.0, 25.0];
        let light = [-30.0, -60.0, 10.0];
        let radius = 1.0;

        let screen = [-0.5, -0.5, 0.0];
        let xadd = 1.0 / canvas.width;
        let yadd = 1.0 / canvas.height;

        let delta = [0.0, 0.0, 0.0];
        let help = [0.0, 0.0, 0.0];

        let x1 = [0.0, 0.0, 0.0];
        let n1 = [0.0, 0.0, 0.0];
        let l1 = [0.0, 0.0, 0.0];
        let v1 = [0.0, 0.0, 0.0];
        let color = [0.0, 0.0, 0.0];

        let ofs = 0;
        for (let y = 0; y < canvas.height; y++) {

            screen[0] = -0.5
            for (let x = 0; x < canvas.width; x++) {

                this.minus (delta, screen, eye);
                this.minus (help, eye, sphereMid);
                let divisor = this.lengthNoSqrt (delta);

                let p = this.scalar (delta, help) / divisor;
                let q = (this.lengthNoSqrt (help) - Math.pow (radius * 3.0, 2.0)) / divisor;

                let r, g, b;
                let h = (p * p) - q;
                if (h > 0.0) {                      // Strahl durchstößt Kugel
                    let sh = Math.sqrt (h);
                    let t1 = -p + sh;
                    let t2 = -p - sh;
                    if (t2 < t1) {                  // t1/x1 immer näher zum Betrachter
                        t1 = t2;
                    }
                    x1[0] = eye[0] + t1 * delta[0]; // Intersection
                    x1[1] = eye[1] + t1 * delta[1];
                    x1[2] = eye[2] + t1 * delta[2];

                    this.minus (n1, x1, sphereMid); // Normale
                    this.normalize (n1);

                    this.minus (l1, light, x1);     // Vektor vom Punkt auf der Kugel zur Lichtquelle
                    this.normalize (l1);

                    this.minus (v1, eye, x1);        // Vektor vom Punkt auf der Kugel zum Betrachter
                    this.normalize (v1);

                    this.material.phong(color, l1, n1, v1);
                    r = color[0] * 255;
                    g = color[1] * 255;
                    b = color[2] * 255;
                } else {
                    r = 250;
                    g = 250;
                    b = 250;
                }

                pixels[ofs] = r;
                pixels[ofs + 1] = g;
                pixels[ofs + 2] = b;
                pixels[ofs + 3] = 255;
                ofs += 4;
                screen[0] += xadd;
            }
            screen[1] += yadd;
        }
        this.context.putImageData (this.image, 0, 0);
    },
    minus: function(delta, screen, eye) {
        delta[0] = screen[0] - eye[0];
        delta[1] = screen[1] - eye[1];
        delta[2] = screen[2] - eye[2];
    },
    lengthNoSqrt: function(x) {
        return this.scalar (x, x);
    },
    scalar: function(a, b) {
        return a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
    },
    length: function(x) {
        return Math.sqrt (this.scalar (x, x));
    },
    normalize: function(x) {
        let h = 1 / this.length (x);
        x[0] = x[0] * h;
        x[1] = x[1] * h;
        x[2] = x[2] * h;
    }
}

class Material {
    constructor(model) {
        if (model !== 'undefined') {
            this.model = JSON.parse(model);
        } else {
            this.model = {
                "ipl": [0.72,0.78,0.5],   // Intensität Punktlicht
                "ial": [0.0,0.13,1.0],    // Intensität ambientes Licht
                "os": [0.58,0.88,0.27],   // Objekt farbe Schlaglicht
                "od": [1.0,0.63,0.75],    // Objekt diffuse farbe
                "ka": 0.75,               // Koeffizienten
                "kd": 0.9,
                "ks": 1.0,
                "power": 2.0
            };
        }
    }

    phong(ret, l, n, v) {
        let nl=n[0]*l[0]+n[1]*l[1]+n[2]*l[2];

        for (let a=0;a<3;a++) {
            n[a]*=2.0*nl;
            n[a]-=l[a];
            ret[a]=0.0;
        }
        let rvp=Math.pow(n[0]*v[0]+n[1]*v[1]+n[2]*v[2],this.model.power);
        for (let a=0;a<3;a++) {
            ret[a]=this.model.ial[a]*this.model.ka*this.model.od[a]
                  +this.model.ipl[a]*(this.model.kd*this.model.os[a]*nl+this.model.ks*rvp);
            if (ret[a]<0.0) {
                ret[a]=0.0;
            }
            if (ret[a]>1.0) {
                ret[a]=1.0;
            }
        }
    }
}
