//
//  RBStarSlider.m
//  Rezeptbuch
//
//  Created by Robert Böhnke on 12/18/12.
//  Copyright (c) 2012 Robert Böhnke. All rights reserved.
//

#import "RBStarSlider.h"

static NSUInteger const RBStarSliderStarWidth   = 36;
static NSUInteger const RBStarSliderStarHeight  = 36;
static NSUInteger const RBStarSliderStarPadding = 8;

@interface RBStarSlider ()

- (UIImage *)fullStar;
- (UIImage *)halfStar;
- (UIImage *)emptyStar;

- (CGFloat)ratingForPoint:(CGPoint)point;

@end

@implementation RBStarSlider

#pragma mark Lifecycle

- (id)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        self.rating = 4.5;
        self.backgroundColor = [UIColor colorWithWhite:0.9 alpha:1];
    }
    return self;
}

#pragma mark - Properties

- (void)setRating:(CGFloat)rating
{
    // Round to nearest half
    _rating = round(rating * 2) / 2;
    _rating = fmaxf(1, fminf(_rating, 5));

    [self sendActionsForControlEvents:UIControlEventValueChanged];
    [self setNeedsDisplay];
}

#pragma mark - UIView

- (void)drawRect:(CGRect)rect
{
    [super drawRect:rect];

    CGFloat requiredWidth  = 5 * RBStarSliderStarWidth + 4 * RBStarSliderStarPadding;

    CGFloat x = (CGRectGetWidth(self.bounds) - requiredWidth) / 2;
    CGFloat y = (CGRectGetHeight(self.bounds) - RBStarSliderStarHeight) / 2;

    for (NSUInteger i = 0; i < 5; i++) {
        UIImage *image;
        if (self.rating <= i) {
            image = self.emptyStar;
        } else if (self.rating <= i + 0.5) {
            image = self.halfStar;
        } else {
            image = self.fullStar;
        }

        [image drawAtPoint:CGPointMake(x, y)];

        x += RBStarSliderStarWidth + RBStarSliderStarPadding;
    }
}

#pragma mark - UIControl

- (BOOL)beginTrackingWithTouch:(UITouch *)touch withEvent:(UIEvent *)event
{
    self.rating = [self ratingForPoint:[touch locationInView:self]];

    return true;
}

- (BOOL)continueTrackingWithTouch:(UITouch *)touch withEvent:(UIEvent *)event
{
    self.rating = [self ratingForPoint:[touch locationInView:self]];

    return true;
}

#pragma mark - RBStarSlider

- (CGFloat)ratingForPoint:(CGPoint)point
{
    CGFloat requiredWidth = 5 * RBStarSliderStarWidth + 4 * RBStarSliderStarPadding;
    CGFloat leftOffset    = (CGRectGetWidth(self.bounds) - requiredWidth) / 2;
    CGFloat fraction      = (point.x - leftOffset) / requiredWidth;

    return 5 * fraction;
}

- (UIImage *)fullStar
{
    return [UIImage imageNamed:@"full-star"];
}

- (UIImage *)halfStar
{
    return [UIImage imageNamed:@"half-star"];
}

- (UIImage *)emptyStar
{
    return [UIImage imageNamed:@"empty-star"];
}

#pragma mark - UIAccessibility

- (BOOL)isAccessibilityElement
{
    return true;
}

- (UIAccessibilityTraits)accessibilityTraits
{
    return UIAccessibilityTraitAdjustable;
}

- (void)accessibilityIncrement
{
    self.rating += 0.5;
}

- (void)accessibilityDecrement
{
    self.rating -= 0.5;
}

- (NSString *)accessibilityLabel
{
    return NSLocalizedString(@"Bewertung", nil);
}

- (NSString *)accessibilityValue
{
    if (self.rating == 1) {
        return NSLocalizedString(@"Ein Stern", nil);
    } else {
        NSString *language = [[NSBundle mainBundle] preferredLocalizations][0];

        NSNumberFormatter *formatter = [[NSNumberFormatter alloc] init];
        formatter.locale = [[NSLocale alloc] initWithLocaleIdentifier:language];
        formatter.numberStyle = NSNumberFormatterDecimalStyle;

        NSString *string = [formatter stringFromNumber:@(self.rating)];
        return [NSString stringWithFormat:NSLocalizedString(@"%@ Sterne", nil), string];
    }
}

@end
