//
//  CTPMasterViewController.m
//  PhotoViewer
//
//  Created by Richard Kurz on 12.06.13.
//  Copyright (c) 2013 Nobody. No rights reserved.
//

#define CPT_USE_ASPECT_SIZE 0

#import "CTPMasterViewController.h"
#import "CTPDetailViewController.h"
#import "CTPCollectionViewCell.h"
#import "CTPSupplementaryView.h"
#import "CTPCollectionViewLayout.h"

#import <AssetsLibrary/AssetsLibrary.h>


@implementation CTPMasterViewController
{
  dispatch_queue_t thumbnailQueue;

  NSArray* layouts;
  ALAssetsLibrary* library;
  
  NSArray* assetData;
  NSArray* groups;
}


- (void) loadPhotos
{
  NSMutableArray* localAssetData = [NSMutableArray new];
  NSMutableArray* localGroups = [NSMutableArray new];
  
  ALAssetsGroupEnumerationResultsBlock assetEnumerator = ^(ALAsset* asset, NSUInteger index, BOOL* stop)
  {
    if (asset)
    {
#if CPT_USE_ASPECT_SIZE
      UIImage* thumb = [UIImage imageWithCGImage: [asset aspectRatioThumbnail]];
      NSValue* thumbSize =  [NSValue valueWithCGSize: thumb.size];
      NSDictionary* data = @{ @"asset": asset, @"size": thumbSize };
#else
      NSDictionary* data = @{ @"asset": asset };
#endif
      
      [[localAssetData lastObject] addObject: data];
    }
  };
  
  ALAssetsLibraryGroupsEnumerationResultsBlock groupEnumerator = ^(ALAssetsGroup* group, BOOL* stop)
  {
    if (group)
    {
      [group setAssetsFilter: [ALAssetsFilter allPhotos]];
      
      if (group.numberOfAssets)
      {
        [localAssetData addObject: [NSMutableArray new]];
        [localGroups addObject: group];
        
        [group enumerateAssetsUsingBlock: assetEnumerator];
      }
    }
    else
    {
      dispatch_async(dispatch_get_main_queue(), ^{
        assetData = localAssetData;
        groups = localGroups;
        [self.collectionView reloadData];
      });
    }
  };
  
  dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
    [library enumerateGroupsWithTypes: ALAssetsGroupAlbum | ALAssetsGroupSavedPhotos
                           usingBlock: groupEnumerator
                         failureBlock: ^(NSError* error) {
                           NSLog(@"Fehler %@", [error localizedDescription]);
                         }];
  });
}


- (void) assetsLibraryDidChange: (NSNotification*) changeNotification
{
  [self loadPhotos];
}


- (void) viewDidLoad
{
  [super viewDidLoad];

  thumbnailQueue = dispatch_queue_create("photoviewer.thumbnail.loading.qeue", NULL);
  
  UICollectionViewFlowLayout* firstLayout = (UICollectionViewFlowLayout*)[self.collectionView collectionViewLayout];
  CTPCollectionViewLayout* secondLayout = [CTPCollectionViewLayout new];
  secondLayout.itemSize = firstLayout.itemSize;
  layouts = @[firstLayout, secondLayout];
  
  library = [ALAssetsLibrary new];
  [[NSNotificationCenter defaultCenter] addObserver: self
                                           selector: @selector(assetsLibraryDidChange:)
                                               name: ALAssetsLibraryChangedNotification
                                             object: library];
  
  [self loadPhotos];
}


- (void) dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver:self];
}


#pragma mark - UICollectionViewDataSource Protocol


- (NSInteger) numberOfSectionsInCollectionView: (UICollectionView*) collectionView
{
  return [assetData count];
}


- (NSInteger) collectionView: (UICollectionView*) view numberOfItemsInSection: (NSInteger) section;
{
  if (!assetData) return 0;
  
  return [assetData[section] count];
}


- (UICollectionViewCell*) collectionView: (UICollectionView*) cv cellForItemAtIndexPath: (NSIndexPath*) indexPath;
{
  CTPCollectionViewCell* cell = [cv dequeueReusableCellWithReuseIdentifier: @"cellID" forIndexPath: indexPath];
  
  cell.label.text = [NSString stringWithFormat: @"(%ld, %ld)", (long)indexPath.section, (long)indexPath.item];
  cell.imageView.image = nil;
  
  id data = assetData[indexPath.section][indexPath.item];
  ALAsset* asset = [data valueForKey: @"asset"];

  dispatch_async(thumbnailQueue, ^{
#if CPT_USE_ASPECT_SIZE
    UIImage* img = [UIImage imageWithCGImage: [asset aspectRatioThumbnail]];
#else
    UIImage* img = [UIImage imageWithCGImage: [asset thumbnail]];
#endif
    dispatch_sync(dispatch_get_main_queue(), ^{
      CTPCollectionViewCell* cell = (CTPCollectionViewCell*)[self.collectionView cellForItemAtIndexPath: indexPath];
      if (cell) cell.imageView.image = img;
    });
  });
  
  return cell;
}


- (UICollectionReusableView*) collectionView: (UICollectionView*) collectionView viewForSupplementaryElementOfKind: (NSString*) kind atIndexPath: (NSIndexPath*) indexPath
{
  CTPSupplementaryView* header = nil;
  
  if ([kind isEqual: UICollectionElementKindSectionHeader])
  {
    header = [collectionView dequeueReusableSupplementaryViewOfKind: kind
                                                withReuseIdentifier: @"headerID"
                                                       forIndexPath: indexPath];
    
    header.label.text = [groups[indexPath.section] valueForProperty: ALAssetsGroupPropertyName];
  }
  
  return header;
}


#pragma mark - UICollectionViewDelegateFlowLayout Protocol


#if CPT_USE_ASPECT_SIZE

- (CGSize) collectionView: (UICollectionView*) collectionView layout: (UICollectionViewLayout*) collectionViewLayout sizeForItemAtIndexPath: (NSIndexPath*) indexPath
{
  id data = assetData[indexPath.section][indexPath.item];
  return [[data valueForKey: @"size"] CGSizeValue];
}


#endif


#pragma mark - Storyboard


- (void) prepareForSegue: (UIStoryboardSegue*) segue sender: (id) sender
{
  if ([[segue identifier] isEqualToString: @"showDetailView"])
  {
    NSIndexPath* indexPath = [[self.collectionView indexPathsForSelectedItems] objectAtIndex: 0];
    id data = assetData[indexPath.section][indexPath.item];
    ALAsset* asset = [data valueForKey: @"asset"];
    ALAssetRepresentation* assetRep = [asset defaultRepresentation];
    UIImage* image = [UIImage imageWithCGImage: [assetRep fullScreenImage]];
    
    CTPDetailViewController* detailViewController = [segue destinationViewController];
    detailViewController.detailItem = image;
  }
}


- (IBAction) switchLayout: (UISegmentedControl*) sender
{
  [self.collectionView setCollectionViewLayout: layouts[sender.selectedSegmentIndex] animated: YES];
  [self.collectionView setContentOffset: CGPointZero animated: YES];
}


@end
