#import "GameViewController.h"

@import Metal;
@import simd;
@import QuartzCore.CAMetalLayer;

@interface GameViewController ()

@property (nonatomic) id<MTLBuffer> buffer;
@property (nonatomic) id<MTLCommandQueue> commandQueue;
@property (nonatomic) CADisplayLink* displayLink;
@property (nonatomic) CAMetalLayer* metalLayer;
@property (nonatomic) id<MTLRenderPipelineState> pipelineState;
@property (nonatomic) id<MTLBuffer> vertexBuffer;

@end

#pragma mark -

@implementation GameViewController

-(BOOL)prefersStatusBarHidden {
    return YES;
}

#pragma mark - Setup

-(void)setupData {
    static const float vertexData[9] = {
        0.0, 1.0, 0.0,
        -1.0, -1.0, 0.0,
        1.0, -1.0, 0.0
    };

    self.vertexBuffer = [self.metalLayer.device newBufferWithBytes:vertexData
                                                            length:sizeof(vertexData)
                                                           options:0];
}

-(void)setupLayer {
    self.metalLayer = [CAMetalLayer layer];
    self.metalLayer.device = MTLCreateSystemDefaultDevice();
    self.metalLayer.frame = self.view.bounds;
    self.metalLayer.pixelFormat = MTLPixelFormatBGRA8Unorm;
    [self.view.layer addSublayer:self.metalLayer];
}

-(void)setupRenderLoop {
    self.commandQueue = [self.metalLayer.device newCommandQueue];

    self.displayLink = [CADisplayLink displayLinkWithTarget:self selector:@selector(renderLoop)];
    [self.displayLink addToRunLoop:NSRunLoop.mainRunLoop forMode:NSDefaultRunLoopMode];
}

-(void)setupShaders {
    id<MTLLibrary> library = [self.metalLayer.device newDefaultLibrary];

    MTLRenderPipelineDescriptor* descriptor = [MTLRenderPipelineDescriptor new];
    descriptor.colorAttachments[0].pixelFormat = self.metalLayer.pixelFormat;
    descriptor.fragmentFunction = [library newFunctionWithName:@"lighting_fragment"];
    descriptor.vertexFunction = [library newFunctionWithName:@"lighting_vertex"];

    NSError* error = nil;
    self.pipelineState = [self.metalLayer.device newRenderPipelineStateWithDescriptor:descriptor
                                                                                error:&error];
    if (!self.pipelineState) {
        NSLog(@"Failed to create pipeline state: %@", error.localizedDescription);
        exit(EXIT_FAILURE);
    }
}

-(void)viewDidLoad {
    [super viewDidLoad];

    [self setupLayer];
    [self setupData];
    [self setupShaders];
    [self setupRenderLoop];
}

#pragma mark - Render Loop

-(void)render {
    id<CAMetalDrawable> drawable = [self.metalLayer nextDrawable];

    MTLRenderPassDescriptor* descriptor = [MTLRenderPassDescriptor new];
    descriptor.colorAttachments[0].clearColor = MTLClearColorMake(0.0, 0.0, 0.0, 1.0);
    descriptor.colorAttachments[0].loadAction = MTLLoadActionClear;
    descriptor.colorAttachments[0].texture = drawable.texture;

    id<MTLCommandBuffer> buffer = [self.commandQueue commandBuffer];

    id<MTLRenderCommandEncoder> encoder = [buffer renderCommandEncoderWithDescriptor:descriptor];
    [encoder setRenderPipelineState:self.pipelineState];
    [encoder setVertexBuffer:self.vertexBuffer offset:0 atIndex:0];
    [encoder drawPrimitives:MTLPrimitiveTypeTriangle vertexStart:0 vertexCount:3 instanceCount:1];
    [encoder endEncoding];

    [buffer presentDrawable:drawable];
    [buffer commit];
}

-(void)renderLoop {
    @autoreleasepool {
        [self render];
    }
}

@end
