#import "GameViewController.h"

@import Metal;
@import simd;
@import QuartzCore.CAMetalLayer;
@import GLKit;

/*
 Um ein 3D-Objekt nicht nur mit Farben, sondern mit einem 2D Bild auf der Oberfläche zu versehen, wird
 eine sogenannte Texturierung ("Texture Mapping") verwendet, welches es ermöglich ein Objekt
 detailreicher aussehen zu lassen ohne das eigentliche Modell zu verändern. Zur Texturierung eines Objektes
 müssen die Eingabedaten um Texturkoordinaten für jedes Vertex erweitert werden. Eine Texturkoordinate uv
 hat dabei zwei Komponenten in einem Koordinatensystem von (0,0) für die linke untere Ecke und (1,1)
 für die rechte Obere Ecke. Diese müssen zusätzlich definiert werden, damit jedes Vertex mit dem
 passenden Teil der Textur versehen wird. Typischerweise werden diese Koordinaten von einer
 3D-Modellierungssoftware errechnet.
 */
static const int componentCount = 5;
static const int vertexCount = 36 * componentCount;

@interface GameViewController ()

@property (nonatomic) id<MTLBuffer> buffer;
@property (nonatomic) id<MTLCommandQueue> commandQueue;
@property (nonatomic) CADisplayLink* displayLink;
@property (nonatomic) CAMetalLayer* metalLayer;
@property (nonatomic) id<MTLRenderPipelineState> pipelineState;
@property (nonatomic) id<MTLSamplerState> samplerState; // Konfiguration des Renderers für Texturen
@property (nonatomic) id<MTLTexture> texture; // Textur
@property (nonatomic) id<MTLBuffer> vertexBuffer;

@property (nonatomic) GLKVector3 position;
@property (nonatomic) GLKVector3 rotation;
@property (nonatomic) float scale;
@property (nonatomic) NSTimeInterval start;

@end

#pragma mark -

@implementation GameViewController

-(BOOL)prefersStatusBarHidden {
    return YES;
}

#pragma mark - Setup

-(void)setupData {
    self.position = GLKVector3Make(0.0, 0.0, 0.0);
    self.rotation = GLKVector3Make(0.0, 0.0, 0.0);
    self.scale = 1.0;

    // Vertexdaten um die Texturkoordinaten erweitert
    static const float vertexData[vertexCount] = {
        -1.0f, 1.0f, -1.0f, 1.0f, 0.0f,
        -1.0f, -1.0f, -1.0f, 1.0f, 1.0f,
        1.0f, 1.0f, -1.0f, 0.0f, 0.0f,
        -1.0f, -1.0f, -1.0f, 1.0f, 1.0f,
        1.0f, -1.0f, -1.0f, 0.0f, 1.0f,
        1.0f, 1.0f, -1.0f, 0.0f, 0.0f,

        -1.0f, 1.0f, 1.0f, 0.0f, 0.0f,
        1.0f, 1.0f, 1.0f, 1.0f, 0.0f,
        -1.0f, -1.0f, 1.0f, 0.0f, 1.0f,
        -1.0f, -1.0f, 1.0f, 0.0f, 1.0f,
        1.0f, 1.0f, 1.0f, 1.0f, 0.0f,
        1.0f, -1.0f, 1.0f, 1.0f, 1.0f,

        -1.0f, 1.0f, -1.0f, 1.0f, 1.0f,
        1.0f, 1.0f, 1.0f, 0.0f, 0.0f,
        -1.0f, 1.0f, 1.0f, 1.0f, 0.0f,
        -1.0f, 1.0f, -1.0f, 1.0f, 1.0f,
        1.0f, 1.0f, -1.0f, 0.0f, 1.0f,
        1.0f, 1.0f, 1.0f, 0.0f, 0.0f,

        -1.0f, -1.0f, -1.0f, 1.0f, 0.0f,
        -1.0f, -1.0f, 1.0f, 1.0f, 1.0f,
        1.0f, -1.0f, 1.0f, 0.0f, 1.0f,
        -1.0f, -1.0f, -1.0f, 1.0f, 0.0f,
        1.0f, -1.0f, 1.0f, 0.0f, 1.0f,
        1.0f, -1.0f, -1.0f, 0.0f, 0.0f,

        -1.0f, 1.0f, -1.0f, 0.0f, 0.0f,
        -1.0f, -1.0f, 1.0f, 1.0f, 1.0f,
        -1.0f, -1.0f, -1.0f, 0.0f, 1.0f,
        -1.0f, 1.0f, 1.0f, 1.0f, 0.0f,
        -1.0f, -1.0f, 1.0f, 1.0f, 1.0f,
        -1.0f, 1.0f, -1.0f, 0.0f, 0.0f,

        1.0f, 1.0f, -1.0f, 1.0f, 0.0f,
        1.0f, -1.0f, -1.0f, 1.0f, 1.0f,
        1.0f, -1.0f, 1.0f, 0.0f, 1.0f,
        1.0f, 1.0f, 1.0f, 0.0f, 0.0f,
        1.0f, 1.0f, -1.0f, 1.0f, 0.0f,
        1.0f, -1.0f, 1.0f, 0.0f, 1.0f
    };

    self.vertexBuffer = [self.metalLayer.device newBufferWithBytes:vertexData
                                                            length:sizeof(float) * vertexCount
                                                           options:0];
}

-(void)setupLayer {
    self.metalLayer = [CAMetalLayer layer];
    self.metalLayer.device = MTLCreateSystemDefaultDevice();
    self.metalLayer.frame = self.view.bounds;
    self.metalLayer.pixelFormat = MTLPixelFormatBGRA8Unorm;
    [self.view.layer addSublayer:self.metalLayer];
}

-(void)setupRenderLoop {
    self.commandQueue = [self.metalLayer.device newCommandQueue];

    self.displayLink = [CADisplayLink displayLinkWithTarget:self selector:@selector(renderLoop)];
    [self.displayLink addToRunLoop:NSRunLoop.mainRunLoop forMode:NSDefaultRunLoopMode];

    self.start = self.displayLink.timestamp;
}

// Einstellungen zur Behandlung von Texturen
-(void)setupSamplerState {
    MTLSamplerDescriptor* descriptor = [MTLSamplerDescriptor new];

    /* 
     Die folgenden Properties konfigurieren diverse Optionen zur Kantenglättung und zur Filterung
     der Texturen. Diese verhindern Artefakte die beim Rendering auftreten, benötigen aber anderseits
     wiederum Rechenzeit. Die hier gewählten Optionen benötigen die minimalste Rechenzeit.
     */
    descriptor.minFilter = MTLSamplerMinMagFilterNearest;
    descriptor.magFilter = MTLSamplerMinMagFilterNearest;
    descriptor.mipFilter = MTLSamplerMipFilterNotMipmapped;
    descriptor.maxAnisotropy = 1;
    descriptor.lodMinClamp = 0;
    descriptor.lodMaxClamp = FLT_MAX;

    /*
     Die folgenden Properties konfigurieren das Verhalten für Bildpunkte, die von der Textur nicht
     abgedeckt werden. Mit den gegebenen Einstellungen erhalten diese keine Texturierung.
     */
    descriptor.rAddressMode = MTLSamplerAddressModeClampToEdge;
    descriptor.sAddressMode = MTLSamplerAddressModeClampToEdge;
    descriptor.tAddressMode = MTLSamplerAddressModeClampToEdge;

    /* 
     Gibt an dass die Texturkoordinaten von 0 bis 1 normalisiert sind. Andernfalls ist ihr maximaler
     Wert die Höhe bzw. Breite der Textur. Um von dieser unabhängig zu bleiben ist es üblich mit
     normalisierten Koordinaten zu rechnen.
     */
    descriptor.normalizedCoordinates = YES;

    self.samplerState = [self.metalLayer.device newSamplerStateWithDescriptor:descriptor];
    NSParameterAssert(self.samplerState);
}

-(void)setupShaders {
    id<MTLLibrary> library = [self.metalLayer.device newDefaultLibrary];

    MTLRenderPipelineDescriptor* descriptor = [MTLRenderPipelineDescriptor new];
    descriptor.colorAttachments[0].pixelFormat = self.metalLayer.pixelFormat;
    descriptor.fragmentFunction = [library newFunctionWithName:@"lighting_fragment"];
    descriptor.vertexFunction = [library newFunctionWithName:@"lighting_vertex"];

    NSError* error = nil;
    self.pipelineState = [self.metalLayer.device newRenderPipelineStateWithDescriptor:descriptor
                                                                                error:&error];
    if (!self.pipelineState) {
        NSLog(@"Failed to create pipeline state: %@", error.localizedDescription);
        exit(EXIT_FAILURE);
    }

    self.texture = [self textureFromImageNamed:@"logo"];
}

// Laden und konvertieren eines Bildes in eine Textur mittels CoreGraphics
-(id<MTLTexture>)textureFromImageNamed:(NSString*)imageName {
    // Laden der Textur als UIImage
    UIImage* textureImage = [UIImage imageNamed:imageName];
    NSParameterAssert(textureImage);

    // Erzeugung eines benötigten CGColorSpace Objekts
    CGColorSpaceRef colorSpace = CGColorSpaceCreateDeviceRGB();
    NSParameterAssert(colorSpace);

    uint32_t width = textureImage.size.width;
    uint32_t height = textureImage.size.height;
    uint32_t rowBytes = width * 4;

    // Erzeugung eines neuen Grafik-Kontextes in Größe des Bildes
    CGContextRef context = CGBitmapContextCreate(NULL,
                                                 width,
                                                 height,
                                                 8,
                                                 rowBytes,
                                                 colorSpace,
                                                 (CGBitmapInfo)kCGImageAlphaPremultipliedLast);

    CGColorSpaceRelease(colorSpace);
    NSParameterAssert(context);

    // Zeichnen des Bildes in den Grafik-Kontext
    CGRect bounds = CGRectMake(0.0f, 0.0f, width, height);
    CGContextClearRect(context, bounds);
    CGContextDrawImage(context, bounds, textureImage.CGImage);

    // Konfigaration des Pixelformats für die Textur
    MTLTextureDescriptor *descriptor = [MTLTextureDescriptor texture2DDescriptorWithPixelFormat:MTLPixelFormatRGBA8Unorm width:width height:height mipmapped:NO];

    // Erzeugung eines neuen Texturobjektes
    id<MTLTexture> texture = [self.metalLayer.device newTextureWithDescriptor:descriptor];
    NSParameterAssert(texture);

    // Auslesen der Pixeldaten des Bildes aus dem Grafik-Kontext
    const void *pixels = CGBitmapContextGetData(context);

    // Kopieren der Pixeldaten in das Texturobjekt
    [texture replaceRegion:MTLRegionMake2D(0, 0, width, height)
               mipmapLevel:0
                 withBytes:pixels
               bytesPerRow:rowBytes];

    CGContextRelease(context);

    return texture;
}

-(void)viewDidLoad {
    [super viewDidLoad];

    [self setupLayer];
    [self setupData];
    [self setupSamplerState];
    [self setupShaders];
    [self setupRenderLoop];
}

#pragma mark - Render Loop

-(void)render {
    id<CAMetalDrawable> drawable = [self.metalLayer nextDrawable];

    MTLRenderPassDescriptor* descriptor = [MTLRenderPassDescriptor new];
    //descriptor.colorAttachments[0].clearColor = MTLClearColorMake(0.0, 0.0, 0.0, 1.0);
    descriptor.colorAttachments[0].loadAction = MTLLoadActionClear;
    descriptor.colorAttachments[0].texture = drawable.texture;

    id<MTLCommandBuffer> buffer = [self.commandQueue commandBuffer];

    id<MTLRenderCommandEncoder> encoder = [buffer renderCommandEncoderWithDescriptor:descriptor];
    [encoder setCullMode:MTLCullModeBack];
    [encoder setRenderPipelineState:self.pipelineState];
    [encoder setVertexBuffer:self.vertexBuffer offset:0 atIndex:0];

    // Textur und deren Konfiguration als Eingabe für den Fragment-Shader
    [encoder setFragmentTexture:self.texture atIndex:0];
    [encoder setFragmentSamplerState:self.samplerState atIndex:0];

    GLKMatrix4 projectionMatrix = GLKMatrix4MakePerspective(GLKMathDegreesToRadians(85.0), self.view.bounds.size.width / self.view.bounds.size.height, 0.1, 100.0);

    GLKMatrix4 pyramidModelMatrix = [self modelMatrix];
    pyramidModelMatrix = GLKMatrix4Multiply([self worldModelMatrix], pyramidModelMatrix);

    id<MTLBuffer> uniformBuffer = [self.metalLayer.device newBufferWithLength:sizeof(float) * 16 * 2
                                                                      options:0];
    void* bufferPointer = [uniformBuffer contents];
    memcpy(bufferPointer, pyramidModelMatrix.m, sizeof(float) * 16);
    memcpy(bufferPointer + sizeof(float) * 16, projectionMatrix.m, sizeof(float) * 16);

    [encoder setVertexBuffer:uniformBuffer offset:0 atIndex:1];

    [encoder drawPrimitives:MTLPrimitiveTypeTriangle
                vertexStart:0 vertexCount:vertexCount / componentCount instanceCount:1];
    [encoder endEncoding];

    [buffer presentDrawable:drawable];
    [buffer commit];
}

-(void)renderLoop {
    @autoreleasepool {
        [self render];

        NSTimeInterval time = self.displayLink.timestamp - self.start;
        const float secondsForMove = 6.0;

        self.rotation = GLKVector3Make(sinf(time * 2.0 * M_PI / secondsForMove),
                                       sinf(time * 2.0 * M_PI / secondsForMove),
                                       self.rotation.z);
    }
}

#pragma mark - Matrices

-(GLKMatrix4)modelMatrix {
    GLKMatrix4 matrix = GLKMatrix4Identity;
    matrix = GLKMatrix4Translate(matrix, self.position.x, self.position.y, self.position.z);
    matrix = [self rotateMatrix:matrix byX:self.rotation.x y:self.rotation.y z:self.rotation.z];
    return GLKMatrix4Scale(matrix, self.scale, self.scale, self.scale);
}

-(GLKMatrix4)rotateMatrix:(GLKMatrix4)matrix byX:(float)xAngle y:(float)yAngle z:(float)zAngle {
    matrix = GLKMatrix4Rotate(matrix, xAngle, 1, 0, 0);
    matrix = GLKMatrix4Rotate(matrix, yAngle, 0, 1, 0);
    return GLKMatrix4Rotate(matrix, zAngle, 0, 0, 1);
}

-(GLKMatrix4)worldModelMatrix {
    GLKMatrix4 matrix = GLKMatrix4Identity;
    matrix = GLKMatrix4Translate(matrix, 0.0, 0.0, -7.0);
    return [self rotateMatrix:matrix byX:GLKMathDegreesToRadians(25.0) y:0.0 z:0.0];
}

@end
