// 2020 Richard Kurz, no rights reserved.

#pragma once

enum
{
  kHID_KeyNoEvent         = 0x00,
  kHID_KeyErrorRollOver   = 0x01,
  kHID_KeyPOSTFail        = 0x02,
  kHID_KeyErrorUndefined  = 0x03,

  kHID_KeyEnter           = 0x28,
  kHID_KeyEscape          = 0x29,
  kHID_KeyBackspace       = 0x2A,
  kHID_KeyTab             = 0x2B,
  kHID_KeySpacebar        = 0x2C,

  kHID_KeyCapsLock        = 0x39,
  kHID_KeyScrollLock      = 0x47,
  kHID_KeyNumLock         = 0x53,

  kHID_KeyPrintScreen     = 0x46,
  kHID_KeyPause           = 0x48,

  kHID_KeyF1              = 0x3A,
  kHID_KeyF2              = 0x3B,
  kHID_KeyF3              = 0x3C,
  kHID_KeyF4              = 0x3D,
  kHID_KeyF5              = 0x3E,
  kHID_KeyF6              = 0x3F,
  kHID_KeyF7              = 0x40,
  kHID_KeyF8              = 0x41,
  kHID_KeyF9              = 0x42,
  kHID_KeyF10             = 0x43,
  kHID_KeyF11             = 0x44,
  kHID_KeyF12             = 0x45,

  kHID_KeyInsert          = 0x49,
  kHID_KeyHome            = 0x4A,
  kHID_KeyPageUp          = 0x4B,
  kHID_KeyDelete          = 0x4C,
  kHID_KeyEnd             = 0x4D,
  kHID_KeyPageDown        = 0x4E,
  kHID_KeyRightArrow      = 0x4F,
  kHID_KeyLeftArrow       = 0x50,
  kHID_KeyDownArrow       = 0x51,
  kHID_KeyUpArrow         = 0x52,

  kHID_KeypadSlash        = 0x54,
  kHID_KeypadAsterisk     = 0x55,
  kHID_KeypadHyphen       = 0x56,
  kHID_KeypadPlus         = 0x57,
  kHID_KeypadEnter        = 0x58,
  kHID_Keypad1            = 0x59,
  kHID_Keypad2            = 0x5A,
  kHID_Keypad3            = 0x5B,
  kHID_Keypad4            = 0x5C,
  kHID_Keypad5            = 0x5D,
  kHID_Keypad6            = 0x5E,
  kHID_Keypad7            = 0x5F,
  kHID_Keypad8            = 0x60,
  kHID_Keypad9            = 0x61,
  kHID_Keypad0            = 0x62,
  kHID_KeypadPeriod       = 0x63,
  kHID_KeypadEqualSign    = 0x67,
  kHID_KeypadComma        = 0x85,

  kHID_KeyF13 = 0x68,
  kHID_KeyF14 = 0x69,
  kHID_KeyF15 = 0x6A,
  kHID_KeyF16 = 0x6B,
  kHID_KeyF17 = 0x6C,
  kHID_KeyF18 = 0x6D,
  kHID_KeyF19 = 0x6E,
  kHID_KeyF20 = 0x6F,
  kHID_KeyF21 = 0x70,
  kHID_KeyF22 = 0x71,
  kHID_KeyF23 = 0x72,
  kHID_KeyF24 = 0x73,

  kHID_KeyExecute = 0x74,
  kHID_KeyHelp    = 0x75,
  kHID_KeyMenu    = 0x76,
  kHID_KeySelect  = 0x77,
  kHID_KeyStop    = 0x78,
  kHID_KeyAgain   = 0x79,
  kHID_KeyUndo    = 0x7A,
  kHID_KeyCut     = 0x7B,
  kHID_KeyCopy    = 0x7C,
  kHID_KeyPaste   = 0x7D,
  kHID_KeyFind    = 0x7E,

  kHID_KeyMute        = 0x7F,
  kHID_KeyVolumeUp    = 0x80,
  kHID_KeyVolumeDown  = 0x81,

  kHID_KeyLockingCapsLock   = 0x82,
  kHID_KeyLockingNumLock    = 0x83,
  kHID_KeyLockingScrollLock = 0x84,

  kHID_KeyAlternateErase    = 0x99,
  kHID_KeySysReqOrAttention = 0x9A,
  kHID_KeyCancel            = 0x9B,
  kHID_KeyClear             = 0x9C,
  kHID_KeyPrior             = 0x9D,
  kHID_KeyReturn            = 0x9E,
  kHID_KeySeparator         = 0x9F,
  kHID_KeyOut               = 0xA0,
  kHID_KeyOper              = 0xA1,
  kHID_KeyClearOrAgain      = 0xA2,
  kHID_KeyCrSelOrProps      = 0xA3,
  kHID_KeyExSel             = 0xA4,

  kHID_KeyLeftControl   = 0xE0,
  kHID_KeyLeftShift     = 0xE1,
  kHID_KeyLeftAlt       = 0xE2,
  kHID_KeyLeftGUI       = 0xE3,
  kHID_KeyRightControl  = 0xE4,
  kHID_KeyRightShift    = 0xE5,
  kHID_KeyRightAlt      = 0xE6,
  kHID_KeyRightGUI      = 0xE7,
};

enum
{
  kHID_MediaPlayOrPause         = 0xcd,
  kHID_MediaScanNextTrack       = 0xb5,
  kHID_MediaScanPreviousTrack   = 0xb6,
  kHID_MediaVolumeIncrement     = 0xe9,
  kHID_MediaVolumeDecrement     = 0xea,
  kHID_MediaVolumeMute          = 0xe2,
};

enum
{
  kHID_LedNumLock      = 0x01,
  kHID_LedCapsLock     = 0x02,
  kHID_LedScrollLock   = 0x03,
  kHID_LedCompose      = 0x04,
  kHID_LedKana         = 0x05,
  kHID_LedPower        = 0x06,
  kHID_LedShift        = 0x07,
  kHID_LedDoNotDisturb = 0x08,
};

enum
{
  kHID_ModLeftControl   = 0x01,
  kHID_ModLeftShift     = 0x02,
  kHID_ModLeftAlt       = 0x04,
  kHID_ModLeftGUI       = 0x08,
  kHID_ModRightControl  = 0x10,
  kHID_ModRightShift    = 0x20,
  kHID_ModRightAlt      = 0x40,
  kHID_ModRightGUI      = 0x80,
};

#define MSL            (kHID_ModLeftShift << 8)
#define MAR            (kHID_ModRightAlt << 8)

static const uint16_t codeMapDE[96] PROGMEM =
{
#if USE_MAC_LAYOUT
  0x2c,             // Space
  0x1e | MSL,       // !
  0x1F | MSL,       // "
  0x32,             // #
  0x21 | MSL,       // $
  0x22 | MSL,       // %
  0x23 | MSL,       // &
  0x32 | MSL,       // '
  0x25 | MSL,       // (
  0x26 | MSL,       // )
  0x30 | MSL,       // *
  0x30,             // +
  0x36,             // ,
  0x38,             // -
  0x37,             // .
  0x24 | MSL,       // /
  0x27,             // 0
  0x1e,             // 1
  0x1f,             // 2
  0x20,             // 3
  0x21,             // 4
  0x22,             // 5
  0x23,             // 6
  0x24,             // 7
  0x25,             // 8
  0x26,             // 9
  0x37 | MSL,       // :
  0x36 | MSL,       // ;
  0x35,             // <
  0x27 | MSL,       // =
  0x35 | MSL,       // >
  0x2d | MSL,       // ?
  0x0f | MAR,       // @
  0x04 | MSL,       // A
  0x05 | MSL,       // B
  0x06 | MSL,       // C
  0x07 | MSL,       // D
  0x08 | MSL,       // E
  0x09 | MSL,       // F
  0x0a | MSL,       // G
  0x0b | MSL,       // H
  0x0c | MSL,       // I
  0x0d | MSL,       // J
  0x0e | MSL,       // K
  0x0f | MSL,       // L
  0x10 | MSL,       // M
  0x11 | MSL,       // N
  0x12 | MSL,       // O
  0x13 | MSL,       // P
  0x14 | MSL,       // Q
  0x15 | MSL,       // R
  0x16 | MSL,       // S
  0x17 | MSL,       // T
  0x18 | MSL,       // U
  0x19 | MSL,       // V
  0x1a | MSL,       // W
  0x1b | MSL,       // X
  0x1d | MSL,       // Y
  0x1c | MSL,       // Z
  0x22 | MAR,       // [
  0x24 | MAR | MSL, // \ backslash
  0x23 | MAR,       // ]
  0x64,             // ^
  0x38 | MSL,       // _
  0x2E | MSL,       // `
  0x04,             // a
  0x05,             // b
  0x06,             // c
  0x07,             // d
  0x08,             // e
  0x09,             // f
  0x0a,             // g
  0x0b,             // h
  0x0c,             // i
  0x0d,             // j
  0x0e,             // k
  0x0f,             // l
  0x10,             // m
  0x11,             // n
  0x12,             // o
  0x13,             // p
  0x14,             // q
  0x15,             // r
  0x16,             // s
  0x17,             // t
  0x18,             // u
  0x19,             // v
  0x1a,             // w
  0x1b,             // x
  0x1d,             // y
  0x1c,             // z
  0x24 | MAR,       // {
  0x25 | MAR,       // |
  0x27 | MAR,       // }
  0x11 | MAR,       // ~
  0x4c              // Delete
#else
  0x2c,        // space
  0x1e | MSL,  // !
  0x1f | MSL,  // "
  0x31,        // #
  0x21 | MSL,  // $
  0x22 | MSL,  // %
  0x23 | MSL,  // &
  0x31 | MSL,  // '
  0x25 | MSL,  // (
  0x26 | MSL,  // )
  0x30 | MSL,  // *
  0x30,        // +
  0x36,        // ,
  0x38,        // -
  0x37,        // .
  0x24 | MSL,  // /
  0x27,        // 0
  0x1e,        // 1
  0x1f,        // 2
  0x20,        // 3
  0x21,        // 4
  0x22,        // 5
  0x23,        // 6
  0x24,        // 7
  0x25,        // 8
  0x26,        // 9
  0x37 | MSL,  // :
  0x36 | MSL,  // ;
  0x64,        // <
  0x27 | MSL,  // =
  0x64 | MSL,  // >
  0x2d | MSL,  // ?
  0x14 | MAR,  // @
  0x04 | MSL,  // A
  0x05 | MSL,  // B
  0x06 | MSL,  // C
  0x07 | MSL,  // D
  0x08 | MSL,  // E
  0x09 | MSL,  // F
  0x0a | MSL,  // G
  0x0b | MSL,  // H
  0x0c | MSL,  // I
  0x0d | MSL,  // J
  0x0e | MSL,  // K
  0x0f | MSL,  // L
  0x10 | MSL,  // M
  0x11 | MSL,  // N
  0x12 | MSL,  // O
  0x13 | MSL,  // P
  0x14 | MSL,  // Q
  0x15 | MSL,  // R
  0x16 | MSL,  // S
  0x17 | MSL,  // T
  0x18 | MSL,  // U
  0x19 | MSL,  // V
  0x1a | MSL,  // W
  0x1b | MSL,  // X
  0x1d | MSL,  // Y
  0x1c | MSL,  // Z
  0x25 | MAR,  // [
  0x2d | MAR,  // \ backslash
  0x26 | MAR,  // ]
  0x35,        // ^
  0x38 | MSL,  // _
  0x2e | MSL,  // `
  0x04,        // a
  0x05,        // b
  0x06,        // c
  0x07,        // d
  0x08,        // e
  0x09,        // f
  0x0a,        // g
  0x0b,        // h
  0x0c,        // i
  0x0d,        // j
  0x0e,        // k
  0x0f,        // l
  0x10,        // m
  0x11,        // n
  0x12,        // o
  0x13,        // p
  0x14,        // q
  0x15,        // r
  0x16,        // s
  0x17,        // t
  0x18,        // u
  0x19,        // v
  0x1a,        // w
  0x1b,        // x
  0x1d,        // y
  0x1c,        // z
  0x24 | MAR,  // {
  0x64 | MAR,  // |
  0x27 | MAR,  // }
  0x30 | MAR,  // ~
  0x4c         // DEL
#endif
};

#undef MSL
#undef MAR


enum
{
  kReportIdKeyboard  = 0x02,
  kReportIdMediaKeys = 0x03
};


#if defined (__AVR_ATtiny85__)
extern "C"
#endif

const char hidReportDescriptor[] PROGMEM =
{
  0x05, 0x01,                    // USAGE_PAGE (Generic Desktop)
  0x09, 0x06,                    // USAGE (Keyboard)
  0xa1, 0x01,                    // COLLECTION (Application)
  0x85, kReportIdKeyboard,       //   REPORT_ID (2)
  0x05, 0x07,                    //   USAGE_PAGE (Keyboard)
  0x19, 0xe0,                    //   USAGE_MINIMUM (Keyboard LeftControl)
  0x29, 0xe7,                    //   USAGE_MAXIMUM (Keyboard Right GUI)
  0x15, 0x00,                    //   LOGICAL_MINIMUM (0)
  0x25, 0x01,                    //   LOGICAL_MAXIMUM (1)
  0x75, 0x01,                    //   REPORT_SIZE (1)
  0x95, 0x08,                    //   REPORT_COUNT (8)
  0x81, 0x02,                    //   INPUT (Data,Var,Abs)

  0x95, 0x01,                    //   REPORT_COUNT (1)
  0x75, 0x08,                    //   REPORT_SIZE (8)
  0x81, 0x03,                    //   INPUT (Cnst,Var,Abs)

  0x95, 0x06,                    //   REPORT_COUNT (6)
  0x75, 0x08,                    //   REPORT_SIZE (8)
  0x15, 0x00,                    //   LOGICAL_MINIMUM (0)
  0x25, 0x73,                    //   LOGICAL_MAXIMUM (115)
  0x05, 0x07,                    //   USAGE_PAGE (Keyboard)
  0x19, 0x00,                    //   USAGE_MINIMUM (Reserved (no event indicated))
  0x29, 0x73,                    //   USAGE_MAXIMUM (Keyboard Application)
  0x81, 0x00,                    //   INPUT (Data,Ary,Abs)

  0x05, 0x08,                    //   USAGE_PAGE (LEDs)
  0x19, 0x01,                    //   USAGE_MINIMUM (NumLock)
  0x29, 0x08,                    //   USAGE_MAXIMUM (DoNotDisturb)
  0x75, 0x01,                    //   REPORT_SIZE (1)
  0x95, 0x08,                    //   REPORT_COUNT (8)
  0x91, 0x02,                    //   OUTPUT (Data,Var,Abs) ; LED report
  0xc0,                          // END_COLLECTION

  0x05, 0x0C,                    // USAGE_PAGE (Consumer Devices)
  0x09, 0x01,                    // USAGE (Consumer Control)
  0xA1, 0x01,                    // COLLECTION (Application)
  0x85, kReportIdMediaKeys,      //   REPORT_ID (1)
  0x19, 0x00,                    //   USAGE_MINIMUM (Unassigned)
  0x2A, 0x3C, 0x02,              //   USAGE_MAXIMUM
  0x15, 0x00,                    //   LOGICAL_MINIMUM (0)
  0x26, 0x3C, 0x02,              //   LOGICAL_MAXIMUM
  0x95, 0x01,                    //   REPORT_COUNT (1)
  0x75, 0x10,                    //   REPORT_SIZE (16)
  0x81, 0x00,                    //   INPUT (Data,Ary,Abs)
  0xC0,                          // END_COLLECTION
};


template <typename TConnection>
struct LayoutDE  : public TConnection, public Print
{
  uint16_t utf8State = 0, utf8Code = 0;

  void sendKeyStroke(uint8_t keyStroke, uint8_t modifiers = 0)
  {
    TConnection::sendKeyPress(keyStroke, modifiers);
    TConnection::sendKeyPress(0, 0);
  }

  void sendMediaKeyStroke(uint16_t keyStroke)
  {
    TConnection::sendMediaKeyPress(keyStroke);
    TConnection::sendMediaKeyPress(0);
  }

  bool decodeUTF8(uint8_t b)
  {
    if (!utf8State)
    {
      if (b < 0x80)
      {
        utf8Code = b;
        return true;
      }
      else if (b < 0xc0)
      {
        return false;
      }
      else if (b < 0xc2)
      {
        goto error;
      }
      else if (b < 0xe0)
      {
        utf8State = 1;
        b &= 0x1f;
      }
      else if (b < 0xf0)
      {
        utf8State = 2;
        b &= 0x0f;
      }
      else if (b < 0xf5)
      {
        utf8State = 3;
        b &= 0x07;
      }
      else
      {
        goto error;
      }
      utf8Code = uint16_t(b) << (utf8State * 6);
      return false;
    }

    if ((b & 0xc0) == 0x80)
    {
      if (utf8State < 1)
        goto error;

      utf8State -= 1;
      utf8Code |= (uint16_t(b) & 0x3f) << (utf8State * 6);
      return utf8State == 0;
    }

error:
    utf8State = 0;
    utf8Code = 0xfffd;
    return true;
  }

  bool utf8ToKeyCode(uint8_t chr, uint8_t* key, uint8_t* mod)
  {
    *key = 0; *mod = 0;

    if (decodeUTF8(chr))
    {
      if (utf8Code >= ' ' && utf8Code <= 0x7f)
      {
        uint16_t cmap = pgm_read_word_near(codeMapDE + (chr - ' '));
        *key = cmap & 0xff;
        *mod = (cmap >> 8) & 0xff;
      }
      else
      {
        switch (utf8Code)
        {
          case '\t': *key = kHID_KeyTab; break;
          case '\n': *key = kHID_KeyEnter; break;
          case '\r': *key = 0; break;

          case 0xdf: *key = 0x2d; break; // ß
          case 0xb4: *key = 0x2e; break; // ´
          case 0xb0: *key = 0x64; *mod = kHID_ModLeftShift; break; // °

          case 0xfc: *key = 0x2f; break; // ü
          case 0xdc: *key = 0x2f; *mod = kHID_ModLeftShift; break;

          case 0xf6: *key = 0x33; break; // ö
          case 0xd6: *key = 0x33; *mod = kHID_ModLeftShift; break;

          case 0xe4: *key = 0x34; break; // ä
          case 0xc4: *key = 0x34; *mod = kHID_ModLeftShift; break;

          default:
            break;
        }
      }
    }

    return *key != 0;
  }


  size_t write(byte chr)
  {
    byte key = 0, mod = 0;

    if (utf8ToKeyCode(chr, &key, &mod))
      sendKeyStroke(key, mod);

    return 1;
  }
};
