// 2020 Richard Kurz, no rights reserved.

#pragma once

#include <BLEDevice.h>
#include <BLEServer.h>
#include <BLEUtils.h>
#include <BLECharacteristic.h>
#include <BLEHIDDevice.h>
#include <HIDTypes.h>
#include <driver/adc.h>


struct BLE_ESP32 : public BLEServerCallbacks, public BLECharacteristicCallbacks
{
  bool connected = false;

  uint8_t keyReport[8] = {0};

  BLEHIDDevice* hid;
  BLECharacteristic* inputKeyboard;
  BLECharacteristic* outputKeyboard;
  BLECharacteristic* inputMediaKeys;

  const char* deviceManufacturer;
  const char* deviceName;


  BLE_ESP32(const char* name = "ESP32 BLE Keyboard", const char* manufacturer = "Espressif")
  {
    deviceName = name;
    deviceManufacturer = manufacturer;
  }


  void begin()
  {
    BLEDevice::init(deviceName);

    BLEServer* server = BLEDevice::createServer();
    hid = new BLEHIDDevice(server);

    hid->reportMap((uint8_t*)hidReportDescriptor, sizeof(hidReportDescriptor));
    hid->manufacturer()->setValue(deviceManufacturer);
    hid->pnp(0x02, 0xe502, 0xa111, 0x0210);
    hid->hidInfo(0x00, 0x01);

    inputKeyboard = hid->inputReport(kReportIdKeyboard);
    outputKeyboard = hid->outputReport(kReportIdKeyboard);
    inputMediaKeys = hid->inputReport(kReportIdMediaKeys);

    outputKeyboard->setCallbacks(this);
    server->setCallbacks(this);

    hid->startServices();

    BLEAdvertising* advertising = server->getAdvertising();
    advertising->addServiceUUID(hid->hidService()->getUUID());
    advertising->setAppearance(HID_KEYBOARD);
    advertising->start();
  }


  void delay(unsigned long ms)
  {
    ::delay(ms);
  }


  bool isConnected(void)
  {
    return connected;
  }


  void sendKeyPress(uint8_t keyPress, uint8_t modifiers)
  {
    keyReport[0] = modifiers;
    keyReport[1] = 0;
    keyReport[2] = keyPress;
    keyReport[3] = 0;
    keyReport[4] = 0;
    keyReport[5] = 0;
    keyReport[6] = 0;
    keyReport[7] = 0;

    inputKeyboard->setValue(keyReport, 8);
    inputKeyboard->notify();
  }


  void sendMediaKeyPress(uint16_t keyPress)
  {
    keyReport[0] = keyPress & 0xff;
    keyReport[1] = keyPress << 8;

    inputMediaKeys->setValue(keyReport, 2);
    inputMediaKeys->notify();
  }


  void onWrite(BLECharacteristic* bc)
  {
    uint8_t* value = (uint8_t*)(bc->getValue().c_str());
    Serial.print("LED: ");
    Serial.println(*value, 2);
  }


  void onConnect(BLEServer* server)
  {
    connected = true;
    Serial.println("Connected!");

#if 0 // hack 
    static unsigned nc = 0;
    if (++nc <= 2) server->disconnect(0);
#endif
  }


  void onDisconnect(BLEServer* server)
  {
    connected = false;
    Serial.println("Disconnected!");
  }
};


LayoutDE<BLE_ESP32> kbdOut;
