'use strict';

// Zunächst gilt es, die benötigten Module zu laden, die entweder in Node.js
// serienmäßig enthalten sind (http und path) oder die über npm nachinstalliert
// wurden.
const bodyParser = require('body-parser');
const express = require('express');
const http = require('http');
const levels = require('./levels.json');
const path = require('path');
const { v4: uuid } = require('uuid');

// Das games-Objekt fungiert als In-Memory-Datenbank. Das vereinfacht zum einen
// das Testen, da man nach jedem Neustart des Servers wieder mit einer leeren
// "Datenbank" beginnt. Zum anderen trägt es dazu bei, das Beispiel einfach zu
// halten, da man nicht auch noch eine externe Datenbank wie MongoDB, MySQL oder
// ähnlich anbinden muss.
const games = {};

// Im nächsten Schritt initialisert man die API als Express-Anwendung …
const api = express();

// … und aktiviert das Ausliefern des Clients sowie das Durchsuchen der
// eingehenden Anfragen nach JSON-Daten, die dann im Falle des Falles geparsed
// werden.
api.use('/', express.static(path.join(__dirname, 'client')));
api.use(bodyParser.json());

// Das erste Command, start-game, dient dazu, eine neue Partie des Spiels zu
// starten.
api.post('/start-game', (req, res) => {
  // Zunächst generiert man eine eindeutige ID für das Spiel, …
  const gameId = uuid();

  // … initialisiert den Datenbankeintrag für diese Partie mit den gewünschten
  // Daten …
  games[gameId] = { level: 1, isWon: false };

  // … und meldet dem Client die erzeugte ID zurück, damit die Spielerin oder
  // der Spieler im weiteren Verlauf auf die neu erzeugte Partie zugreifen kann.
  res.status(200).json({ gameId });
});

// Das zweite Command, make-guess, dient dazu, dass die Spielerin oder der
// Spieler eine Vermutung für ein gestelltes Rätsel äußern kann.
api.post('/make-guess/:id', (req, res) => {
  // Dazu liest man zunächst die ID der gewünschten Partie, auf die sich die
  // Vermutung bezieht, aus der URL aus und sucht sich den passenden Eintrag in
  // der "Datenbank" heraus.
  const gameId = req.params.id;
  const game = games[gameId];

  // Anschließend prüft man, ob die angegebene Partie überhaupt existiert und
  // ob sie noch läuft oder bereits gewonnen wurde. Je nach Situation kann man
  // die weitere Verarbeitung hier bereits abbrechen, allerdings mit der
  // jeweiligen Situation angepassten, unterschiedlichen Fehlercodes (der HTTP-
  // Statuscode 404 bedeutet "nicht gefunden", 400 hingegen bedeutet, dass die
  // Anfrage ungültig war).
  if (game === undefined) {
    return res.status(404).end();
  }
  if (game.isWon) {
    return res.status(400).end();
  }

  // Nachdem man die Partie validiert hat, kann man das aktuelle Level und die
  // tatsächliche Lösung auslesen.
  const currentLevel = game.level;
  const { solution } = levels[currentLevel - 1];
  const { guess } = req.body;

  // Um zu vermeiden, dass sich die Vermutung nur in der Groß-Kleinschreibung
  // von der tatsächlichen Lösung unterscheidet, konvertiert man beide komplett
  // in Kleinbuchstaben.
  const normalizedGuess = guess.trim().toLowerCase();
  const normalizedSolution = solution.toLowerCase();

  // Falls die Vermutung falsch ist, kann man die Verarbeitung hier beenden.
  if (normalizedGuess !== normalizedSolution) {
    return res.status(400).end();
  }

  // Andernfalls berechnet man das nächste Level …
  const nextLevel = currentLevel + 1;

  // … und überprüft, ob es überhaupt noch ein nächstes Level gibt. Falls nicht,
  // ist die Partie gewonnen und das Spiel kann beendet werden.
  if (nextLevel > levels.length) {
    games[gameId] = { level: undefined, isWon: true };

    return res.status(200).end();
  }

  // Andernfalls wird der Eintrag in der Datenbank auf das nächste Level
  // aktualisiert.
  games[gameId] = { level: nextLevel, isWon: false };
  res.status(200).end();
});

// Die erste Query, current-state, liefert den Status einer konkreten Partie an
// den Client zurück.
api.get('/current-state/:id', (req, res) => {
  // Erneut geht es zunächst darum, die Partie zu validieren und zu laden.
  const gameId = req.params.id;
  const game = games[gameId];

  if (game === undefined) {
    return res.status(404).end();
  }
  if (game.isWon) {
    return res.status(200).json({ isWon: true });
  }

  // Anschließend bestimmt man das aktuelle Level und das dazugehörige Rätsel, …
  const currentLevel = game.level;
  const { riddle } = levels[currentLevel - 1];

  // … um dann beides an den Client zu senden.
  res.status(200).json({
    isWon: false,
    level: currentLevel,
    riddle
  });
});

// Die zweite Query, highscore, dient dazu, den aktuellen Highscore zu
// ermitteln.
api.get('/highscore', (req, res) => {
  // Zunächst geht man davon aus, dass noch kein Highscore erreicht wurde.
  let highscore = 0;

  // Anschließend durchläuft man alle Partien und liest deren bislang höchstes
  // erreichtes Level aus, zieht eins ab (denn das aktuelle Level wurde ja noch
  // nicht bestanden), und nimmt dann den höheren Wert: Entweder den bisher
  // höchsten gefundenen Wert oder den neuen, falls dieser höher liegt.
  for (const game of Object.values(games)) {
    highscore = Math.max(highscore, game.level - 1);
  }

  // Abschließend gibt man den Highscore noch an den Client zurück.
  res.status(200).json({ highscore });
});

// Nun gilt es nur noch, einen HTTP-Server zu erzeugen, ihn mit der Auslieferung
// der API zu beauftragen …
const server = http.createServer(api);

// … und ihn auf Port 3000 zu starten.
server.listen(3000);
