//
//  ContentView.swift
//  HeiseRealityKit
//
//  Created by Gero Gerber on 19.11.21.
//

import Combine
import SwiftUI
import RealityKit

class MyModel: ObservableObject {
    public enum Scenes {
        case firstScene
        case secondScene
    }
    
    @Published var currentScene: Scenes = .firstScene
    var triggerAppEvent = PassthroughSubject<Void, Never>()
}

struct ContentView : View {
    @StateObject var myModel = MyModel()

    var body: some View {
        ZStack {
            ARViewContainer()
                .environmentObject(myModel)
                .edgesIgnoringSafeArea(.all)
            VStack {
                Spacer()
                Button {
                    myModel.triggerAppEvent.send()
                } label: {
                    Text("Send Event")
                }
                .buttonStyle(DefaultButtonStyle())
                .padding()
                Button {
                    myModel.currentScene = myModel.currentScene == .firstScene ? .secondScene : .firstScene
                } label: {
                    Text("Toggle Scene")
                }
                .buttonStyle(DefaultButtonStyle())
                .padding()
            }
        }
    }
}

struct ARViewContainer: UIViewRepresentable {
    @EnvironmentObject var myModel: MyModel

    func makeCoordinator() -> Coordinator {
        Coordinator(myModel: myModel)
    }

    func makeUIView(context: Context) -> ARView {
        let arView = ARView(frame: .zero)
        context.coordinator.arView = arView
        return arView
    }

    func updateUIView(_ uiView: ARView, context: Context) {
        context.coordinator.loadScene()
    }
    
    class Coordinator {
        let myModel: MyModel
        var arView: ARView?
        var currentScene: Entity?
        var subscriptions = Set<AnyCancellable>()
        
        init(myModel: MyModel) {
            self.myModel = myModel
            
            myModel.triggerAppEvent
                .sink {
                    self.triggerEventInScene()
                }
                .store(in: &subscriptions)
        }
        
        public func loadScene() {
            if let currentScene = currentScene as? HasAnchoring,
               let arView = arView {
                arView.scene.removeAnchor(currentScene)
            }
            
            switch myModel.currentScene {
                case .firstScene:
                    let scene = try! Heise.loadMyFirstScene()
                    scene.actions.myBehaviourEvent.onAction = myCustomEventHandler(_:)
                    currentScene = scene
                case .secondScene:
                    currentScene = try! Heise.loadMySecondScene()
            }
            
            if let anchor = currentScene as? HasAnchoring,
               let arView = arView {
                arView.scene.anchors.append(anchor)
            }
        }

        private func triggerEventInScene() {
            (currentScene as? Heise.MyFirstScene)?.notifications.myAppEvent.post()
        }
        
        private func myCustomEventHandler(_ entity: Entity?) {
            print("Event triggered for entity \(entity?.name ?? "Unknown")")
        }
    }
}

#if DEBUG
struct ContentView_Previews : PreviewProvider {
    static var previews: some View {
        ContentView()
    }
}
#endif
