//
//  Copyright © 2021 app|tects All rights reserved.
//

import Combine
import Foundation
import RealityKit
import SwiftUI

class AssetModel: ObservableObject {
    @Published public var allAssetsLoaded: Bool = false

    public lazy var dissolveTexture: CustomMaterial.Texture? = {
        if let dissolveTexture = try? TextureResource.load(named: "dissolveTexture.jpg") {
            return CustomMaterial.Texture(dissolveTexture)
        }

        return nil
    }()

    public static let basketBallEntityName = "basketball_entity"
    public static let triggerVolumeEntityName = "trigger_entity"

    private enum EntityType {
        case trashCan
        case basketball
    }

    private var surfaceShader: CustomMaterial.SurfaceShader
    private var geometryModifier: CustomMaterial.GeometryModifier

    private var entities: [EntityType: ModelEntity] = .init()
    private let assetNames: [EntityType: String] = [.trashCan: "trash_can", .basketball: "basketball"]

    private static let bottomRingSize: Float = 0.36
    private static let topRingSize: Float = 0.53
    private static let shapeThickness: Float = 0.01
    private static let height: Float = 0.52
    private static let rotation = Float(Angle(degrees: 80).radians)

    init() {
        guard let device = MTLCreateSystemDefaultDevice() else {
            fatalError("Error creating default metal device.")
        }

        guard let library = device.makeDefaultLibrary() else {
            fatalError("Error loading library!")
        }

        surfaceShader = CustomMaterial.SurfaceShader(named: "dissolveShader", in: library)
        geometryModifier = CustomMaterial.GeometryModifier(named: "spikeShader", in: library)

        loadEntity(with: assetNames[.basketball]!, entityType: .basketball) { basketball in
            self.configureBasketballEntity(basketball: basketball)
        }

        loadEntity(with: assetNames[.trashCan]!, entityType: .trashCan) { trashCan in
            trashCan.physicsBody = PhysicsBodyComponent(massProperties: .default, material: nil, mode: .static)

            let bottomShape = ShapeResource.generateBox(size: simd_float3(Self.bottomRingSize, Self.shapeThickness, Self.bottomRingSize))
            let rightSideShape = ShapeResource.generateBox(size: simd_float3(Self.height, Self.shapeThickness, Self.topRingSize))
                .offsetBy(rotation: simd_quatf(angle: Self.rotation, axis: simd_float3(0, 0, 1)), translation: simd_float3(Self.bottomRingSize / 2, Self.height / 2, 0))
            let leftSideShape = ShapeResource.generateBox(size: simd_float3(Self.height, Self.shapeThickness, Self.topRingSize))
                .offsetBy(rotation: simd_quatf(angle: -Self.rotation, axis: simd_float3(0, 0, 1)), translation: simd_float3(-Self.bottomRingSize / 2, Self.height / 2, 0))
            let backSideShape = ShapeResource.generateBox(size: simd_float3(Self.topRingSize, Self.shapeThickness, Self.height))
                .offsetBy(rotation: simd_quatf(angle: Self.rotation, axis: simd_float3(1, 0, 0)), translation: simd_float3(0, Self.height / 2, -Self.bottomRingSize / 2))
            let frontSideShape = ShapeResource.generateBox(size: simd_float3(Self.topRingSize, Self.shapeThickness, Self.height))
                .offsetBy(rotation: simd_quatf(angle: -Self.rotation, axis: simd_float3(1, 0, 0)), translation: simd_float3(0, Self.height / 2, Self.bottomRingSize / 2))
            trashCan.collision = CollisionComponent(shapes: [bottomShape, leftSideShape, rightSideShape, backSideShape, frontSideShape])
        }
    }

    private func configureBasketballEntity(basketball: ModelEntity) {
        let bounds = basketball.visualBounds(relativeTo: nil)
        let radius = ((bounds.max - bounds.min).x / 2)
        let offset = bounds.center
        let ballShape = ShapeResource.generateSphere(radius: radius).offsetBy(translation: offset)

        basketball.collision = CollisionComponent(shapes: [ballShape], mode: .default, filter: CollisionFilter(group: .all, mask: .all))

        var physicsBody = PhysicsBodyComponent(shapes: [ballShape], mass: 0.624)
        physicsBody.material = PhysicsMaterialResource.generate(staticFriction: 0.41, dynamicFriction: 0.41, restitution: 0.88)
        physicsBody.mode = .dynamic

        basketball.physicsBody = physicsBody

        guard var modelComponent = basketball.components[ModelComponent.self] as? ModelComponent else {
            return
        }

        guard let customMaterials = try? modelComponent.materials.map({ material -> CustomMaterial in
            var customMaterial = try CustomMaterial(from: material, surfaceShader: self.surfaceShader, geometryModifier: self.geometryModifier)
            customMaterial.custom.value[0] = 0
            customMaterial.custom.texture = self.dissolveTexture
            return customMaterial
        }) else {
            return
        }

        modelComponent.materials = customMaterials
        basketball.components[ModelComponent.self] = modelComponent

        basketball.name = Self.basketBallEntityName
    }

    public func cloneTrashCanEntity() -> ModelEntity? {
        entities[.trashCan]?.clone(recursive: true)
    }

    public func cloneBasketballEntity() -> ModelEntity? {
        if let entity = entities[.basketball]?.clone(recursive: true) {
            entity.components[BasketballComponent.self] = BasketballComponent()
            return entity
        }

        return nil
    }

    public func createGroundPlane() -> Entity {
        let planeMesh = MeshResource.generatePlane(width: 5, depth: 5)
        let planeEntity = ModelEntity(mesh: planeMesh, materials: [OcclusionMaterial(receivesDynamicLighting: false)])
        planeEntity.physicsBody = PhysicsBodyComponent(massProperties: .default, material: PhysicsMaterialResource.generate(friction: 0.5, restitution: 0.01), mode: .static)
        planeEntity.collision = CollisionComponent(shapes: [.generateBox(width: 5, height: 0.0001, depth: 5)])
        return planeEntity
    }

    public func createTriggerVolume() -> Entity {
        let triggerVolume = TriggerVolume(shape: ShapeResource.generateBox(size: simd_float3(Self.bottomRingSize / 2.0, 0.1, Self.bottomRingSize / 2.0)), filter: CollisionFilter(group: .all, mask: .all))
        triggerVolume.name = Self.triggerVolumeEntityName
        return triggerVolume
    }

    private func loadEntity(with name: String, entityType: EntityType, completion: @escaping (ModelEntity) -> Void) {
        var cancellable: AnyCancellable?

        cancellable = ModelEntity.loadModelAsync(named: name)
            .sink(receiveCompletion: { error in
                print("Unexpected error: \(error)")
                cancellable?.cancel()
            }, receiveValue: { entity in
                DispatchQueue.main.async {
                    completion(entity)
                    self.entities[entityType] = entity
                    self.allAssetsLoaded = self.entities.count == self.assetNames.count
                    cancellable?.cancel()
                }
            })
    }
}
