//
//  Copyright © 2021 app|tects All rights reserved.
//

import ARKit
import Combine
import CoreImage
import CoreImage.CIFilterBuiltins
import MetalKit
import RealityKit
import simd
import SwiftUI

struct GameView: UIViewRepresentable {
    @EnvironmentObject var assetModel: AssetModel

    public static var fullscreen: some View {
        GameView()
            .ignoresSafeArea()
    }

    func updateUIView(_ uiView: UIViewType, context: Context) {}

    func makeCoordinator() -> Coordinator {
        Coordinator(assetModel: assetModel)
    }

    func makeUIView(context: Context) -> ARView {
        BasketballComponent.registerComponent()
        BasketballSystem.registerSystem()

        let view = ARView()
        view.environment.sceneUnderstanding.options.insert([.physics, .occlusion, .receivesLighting])
        view.debugOptions = [
            // .showPhysics
            // .showAnchorOrigins,
            // .showSceneUnderstanding
        ]

        context.coordinator.view = view
        context.coordinator.setupInitialScene()

        let tapGesture = UITapGestureRecognizer(target: context.coordinator, action: #selector(context.coordinator.viewTap))
        view.addGestureRecognizer(tapGesture)

        let coachingOverlay = ARCoachingOverlayView()
        coachingOverlay.autoresizingMask = [.flexibleWidth, .flexibleHeight]
        coachingOverlay.session = view.session
        coachingOverlay.goal = .horizontalPlane
        view.addSubview(coachingOverlay)

        return view
    }

    class Coordinator: NSObject {
        public var view: ARView?
        private var postProcess: PostProcess?
        private var assetModel: AssetModel
        private var anchor: AnchorEntity?
        private var cameraAnchor: AnchorEntity?
        private var subscriptions: Set<AnyCancellable> = .init()
        private var knownCollisions: Set<UInt64> = .init()
        private var filterEndTime = Date()

        private static let filterDuration: Double = 3

        init(assetModel: AssetModel) {
            self.assetModel = assetModel
        }

        public func setupInitialScene() {
            let anchor = AnchorEntity(plane: .horizontal)

            self.anchor = anchor

            view?.scene.anchors.append(anchor)

            if let trashCan = assetModel.cloneTrashCanEntity() {
                anchor.addChild(trashCan)
            }

            anchor.addChild(assetModel.createTriggerVolume())
            anchor.addChild(assetModel.createGroundPlane())

            cameraAnchor = AnchorEntity(.camera)
            view?.scene.addAnchor(cameraAnchor!)

            view?.scene.subscribe(
                to: CollisionEvents.Began.self
            ) { [weak self] event in
                if let self = self {
                    if event.isCollisionBetweenBasketballAndTrashCan {
                        let basketBallEntity = event.basketBallEntity
                        if !self.knownCollisions.contains(basketBallEntity.id) {
                            self.filterEndTime = Date().addingTimeInterval(Self.filterDuration)
                            self.knownCollisions.insert(basketBallEntity.id)
                        }
                    }
                }
            }
            .store(in: &subscriptions)

            view?.renderCallbacks.prepareWithDevice = { [weak self] device in
                self?.postProcess = PostProcess(device: device)
            }

            view?.renderCallbacks.postProcess = { [weak self] context in
                if let self = self {
                    let progress: Float = Date() > self.filterEndTime ? 0 : 1.0 - (Float(Date().distance(to: self.filterEndTime)) / Float(Self.filterDuration)).clamped(to: 0 ... 1)
                    self.postProcess?.render(context: context, progress: progress)
                }
            }
        }

        private func spawnBasketball() {
            if let cameraAnchor = cameraAnchor,
               let basketball = assetModel.cloneBasketballEntity() {
                cameraAnchor.addChild(basketball)
                basketball.applyImpulse(simd_float3(0, 0, -2.5), at: simd_float3(0, 0.01, 0), relativeTo: cameraAnchor)
            }
        }

        @objc func viewTap() {
            spawnBasketball()
        }
    }
}
