//
//  Copyright © 2021 app|tects All rights reserved.
//

import ARKit
import Combine
import RealityKit
import RoomPlan
import SwiftUI

struct CustomRoomCaptureView: UIViewRepresentable {
    @EnvironmentObject var captureModel: RoomCaptureModel

    func makeUIView(context: Context) -> ARView {
        let view = ARView()
        context.coordinator.setup(view: view)
        return view
    }

    func makeCoordinator() -> CustomRoomCaptureCoordinator {
        CustomRoomCaptureCoordinator(captureModel: captureModel)
    }

    func updateUIView(_ uiView: ARView, context: Context) {}
}

class CustomRoomCaptureCoordinator: NSObject {
    var view: ARView?
    var anchor: AnchorEntity
    var captureSession: RoomCaptureSession
    var captureModel: RoomCaptureModel
    var subscriptions: Set<AnyCancellable> = .init()

    private static var alpha: CGFloat = 0.3

    init(captureModel: RoomCaptureModel) {
        anchor = AnchorEntity()
        anchor.transform = .identity
        captureSession = RoomCaptureSession()
        self.captureModel = captureModel

        super.init()

        captureSession.delegate = self

        captureModel.$isModelVisible.sink { isModelVisible in
            self.anchor.isEnabled = isModelVisible
        }
        .store(in: &subscriptions)

        captureModel.$isCaptureActive.sink { isCaptureActive in
            if isCaptureActive {
                self.removeAllChildrenFromAnchor()

                self.captureSession.run(configuration: RoomCaptureSession.Configuration())
            } else {
                self.captureSession.stop()
            }
        }
        .store(in: &subscriptions)
    }

    func setup(view: ARView) {
        self.view = view
        self.view?.scene.anchors.append(anchor)
        self.view?.session = captureSession.arSession
        captureSession.arSession.delegate = self
    }

    private func addToScene(room: CapturedRoom) {
        createSurfaces(surfaces: room.surfaces)
        createObjects(objects: room.objects)
    }

    private func updateScene(room: CapturedRoom) {
        removeChildrenFromAnchor(roomPlanIds: room.allIdentifiers)

        createSurfaces(surfaces: room.surfaces)
        createObjects(objects: room.objects)
    }

    private func removeFromScene(room: CapturedRoom) {
        removeChildrenFromAnchor(roomPlanIds: room.allIdentifiers)
    }

    private func addHoverInfo(roomPlanId: String, category: String, confidence: String) {
        captureModel.hoverInfos.append(HoverInfo(id: roomPlanId,
                                                 title: category,
                                                 confidenceText: "Confidence: \(confidence)",
                                                 idText: "UUID: \(roomPlanId)",
                                                 screenPosition: .zero,
                                                 isVisible: false))
    }

    private func removeAllChildrenFromAnchor() {
        captureModel.hoverInfos.removeAll()

        let children = anchor.children

        children.forEach { child in
            child.removeFromParent()
        }
    }

    private func removeChildrenFromAnchor(roomPlanIds: [String]) {
        captureModel.hoverInfos.removeAll(where: { roomPlanIds.contains($0.id) })

        anchor.children
            .filter { roomPlanIds.contains($0.name) }
            .forEach { $0.removeFromParent() }
    }

    private func createSurfaces(surfaces: [CapturedRoom.Surface]) {
        surfaces.forEach { surface in
            anchor.addChild(Self.createEntity(surface: surface, color: surface.category.color))
            addHoverInfo(roomPlanId: surface.identifier.uuidString, category: surface.category.description, confidence: surface.confidence.description)
        }
    }

    private func createObjects(objects: [CapturedRoom.Object]) {
        objects.forEach { object in
            anchor.addChild(Self.createEntity(object: object, color: object.category.color))
            addHoverInfo(roomPlanId: object.identifier.uuidString, category: object.category.description, confidence: object.confidence.description)
        }
    }

    private static func createEntity(surface: CapturedRoom.Surface, color: UIColor) -> ModelEntity {
        let surfaceEntity = Self.createEntity(roomPlanId: surface.identifier.uuidString,
                                              dimensions: surface.dimensions,
                                              transform: surface.transform,
                                              color: color,
                                              description: surface.category.description,
                                              textColor: .black)

        surface.completedEdges.forEach { edge in
            let size: Float = 0.01
            var transform = Transform()
            var dimensions = simd_float3(repeating: size)

            switch edge {
                case .top:
                    transform.translation.y = surface.dimensions.y / 2.0
                    dimensions.x = surface.dimensions.x
                case .right:
                    transform.translation.x = surface.dimensions.x / 2.0
                    dimensions.y = surface.dimensions.y
                case .bottom:
                    transform.translation.y = -surface.dimensions.y / 2.0
                    dimensions.x = surface.dimensions.x
                case .left:
                    transform.translation.x = -surface.dimensions.x / 2.0
                    dimensions.y = surface.dimensions.y
                default:
                    break
            }

            surfaceEntity.addChild(Self.createEntity(roomPlanId: surface.identifier.uuidString + "_edge",
                                                     dimensions: dimensions,
                                                     transform: transform.matrix,
                                                     color: .black,
                                                     description: "Edge",
                                                     textColor: .black))
        }

        return surfaceEntity
    }

    private static func createEntity(object: CapturedRoom.Object, color: UIColor) -> ModelEntity {
        Self.createEntity(roomPlanId: object.identifier.uuidString,
                          dimensions: object.dimensions,
                          transform: object.transform,
                          color: color,
                          description: object.category.description,
                          textColor: .black)
    }

    private static func createEntity(roomPlanId: String, dimensions: simd_float3, transform: simd_float4x4, color: UIColor, description: String, textColor: UIColor) -> ModelEntity {
        let planeMesh = MeshResource.generateBox(width: dimensions.x, height: dimensions.y, depth: dimensions.z)
        let modelEntity = ModelEntity(mesh: planeMesh, materials: [UnlitMaterial(color: color)])
        modelEntity.name = roomPlanId
        modelEntity.transform.matrix = transform
        return modelEntity
    }
}

// MARK: - RoomCaptureSessionDelegate

extension CustomRoomCaptureCoordinator: RoomCaptureSessionDelegate {
    func captureSession(_ session: RoomCaptureSession, didAdd room: CapturedRoom) {
        DispatchQueue.main.async {
            self.addToScene(room: room)
        }
    }

    func captureSession(_ session: RoomCaptureSession, didChange room: CapturedRoom) {
        DispatchQueue.main.async {
            self.updateScene(room: room)
        }
    }

    func captureSession(_ session: RoomCaptureSession, didRemove room: CapturedRoom) {
        DispatchQueue.main.async {
            self.removeFromScene(room: room)
        }
    }

    func captureSession(_ session: RoomCaptureSession, didUpdate room: CapturedRoom) {
        DispatchQueue.main.async {
            self.updateScene(room: room)
        }
    }

    func captureSession(_ session: RoomCaptureSession, didProvide instruction: RoomCaptureSession.Instruction) {
        DispatchQueue.main.async {
            self.captureModel.currentInstruction = instruction.description
        }
    }

    func captureSession(_ session: RoomCaptureSession, didStartWith configuration: RoomCaptureSession.Configuration) {}

    func captureSession(_ session: RoomCaptureSession, didEndWith data: CapturedRoomData, error: Error?) {
        Task {
            let roomBuilder = RoomBuilder(options: .beautifyObjects)

            do {
                let capturedRoom = try await roomBuilder.capturedRoom(from: data)

                capturedRoom.export()
            } catch {
                print(error.localizedDescription)
            }
        }
    }
}

// MARK: - ARSessionDelegate

extension CustomRoomCaptureCoordinator: ARSessionDelegate {
    func session(_ session: ARSession, didUpdate frame: ARFrame) {
        guard let scene = UIApplication.shared.connectedScenes.first,
              let sceneDelegate = scene as? UIWindowScene else { return }

        let orientation = sceneDelegate.interfaceOrientation
        let viewDirection = -frame.camera.transform.columns.2

        for i in 0 ..< captureModel.hoverInfos.count {
            var currentHoverInfo = captureModel.hoverInfos[i]

            if let scene = view?.scene,
               let child = scene.findEntity(named: currentHoverInfo.id) as? ModelEntity,
               let viewSize = view?.bounds.size {
                let cameraToChild = simd_float4(child.transform.translation, 0) - frame.camera.transform.columns.3
                let projectedPoint = frame.camera.projectPoint(child.transform.translation, orientation: orientation, viewportSize: viewSize)
                currentHoverInfo.screenPosition = projectedPoint
                currentHoverInfo.isVisible = simd_dot(viewDirection, cameraToChild) > 0
                captureModel.hoverInfos[i] = currentHoverInfo
            }
        }
    }
}
