//
// Agenda.swift
//
// Copyright (c) 2023 by Simon Hartmann, Mark Zimmermann
//

import ActivityKit
import Foundation

class Agenda: ObservableObject {
    @Published var topics: [Topic]
    var activity: Activity<AgendaAttributes>?

    static var empty: Agenda {
        Agenda(topics: [])
    }

    var activeTopic: Topic? {
        topics.first { $0.startDate != nil && $0.endDate == nil }
    }

    var isRunning: Bool {
        topics.first { $0.isActive } != nil
    }

    init(topics: [Topic]) {
        self.topics = topics
    }

    func newTopic() -> Topic {
        let newTopic = Topic.new
        topics.append(newTopic)
        return newTopic
    }

    func addTopic(title: String, hours: Int, minutes: Int) {
        let newTopic = Topic(hours: hours, minutes: minutes, title: title)
        topics.append(newTopic)
    }

    func start() {
        objectWillChange.send()
        topics.forEach { $0.reset() }
        topics.first?.start()
        startOrUpdateLiveActivity()
    }

    func startNextTopic() {
        objectWillChange.send()
        topics.first { $0.isActive }?.end()
        topics.first { $0.isForthcoming }?.start()
        startOrUpdateLiveActivity()
    }

    private func startOrUpdateLiveActivity() {
        do {
            guard let activeTopic else {
                Task { await activity?.end(using: nil, dismissalPolicy: .immediate) }
                return
            }

            if let activity {
                Task { await activity.update(using: AgendaAttributes.ContentState(topic: activeTopic.toLiveActivityModel())) }
            } else {
                activity = try Activity.request(attributes: .init(startDate: .now),
                                                contentState: .init(topic: activeTopic.toLiveActivityModel()))
            }
        } catch {
            debugPrint(error)
        }
    }
}
