//
// AgendaView.swift
//
// Copyright (c) 2023 by Simon Hartmann, Mark Zimmermann
//

import ActivityKit
import SwiftUI

struct AgendaView: View {
    @EnvironmentObject var agenda: Agenda

    var body: some View {
        List {
            ForEach(agenda.topics) { topic in
                HStack {
                    VStack(alignment: .leading) {
                        Text(topic.title)
                            .font(.title3.bold())
                        Text(topic.duration.converted(to: .minutes).formatted())
                            .font(.footnote)
                            .foregroundColor(.gray)
                    }
                    Spacer()
                    VStack(alignment: .trailing) {
                        if let startDate = topic.startDate {
                            Text("Von: \(startDate.formatted(date: .omitted, time: .shortened))")
                                .font(.caption)
                                .foregroundColor(.gray)
                        }

                        if let endDate = topic.endDate {
                            Text("Bis: \(endDate.formatted(date: .omitted, time: .shortened))")
                                .font(.caption)
                                .foregroundColor(.gray)
                        }
                    }
                }
                .padding(10)
                .background(.white)
                .cornerRadius(10)
                .padding(4)
                .listRowSeparator(.hidden)
                .listRowInsets(EdgeInsets(top: 0, leading: 0, bottom: 0, trailing: 0))
                .listRowBackground(Color(white: 0.95))
            }
            .onMove(perform: move)
            .onDelete(perform: delete)
        }
        .padding()
        .listStyle(InsetListStyle())
        .navigationTitle("Agenda")
        .background(Color(white: 0.95))
        .scrollContentBackground(.hidden)
        .toolbar {
            NavigationLink(
                destination: {
                    EditTopicView()
                        .environmentObject(agenda)
                }, label: {
                    Image(systemName: "plus.circle")
                }
            )
            Button(
                action: {
                    agenda.isRunning
                        ? agenda.startNextTopic()
                        : agenda.start()
                },
                label: {
                    agenda.isRunning
                        ? Image(systemName: "forward.fill")
                        : Image(systemName: "play.fill")
                }
            )
            .disabled(agenda.topics.isEmpty)
        }
        .onOpenURL { url in
            guard url.scheme == "de.agendaapp" else { return }

            if url.host() == "next" {
                agenda.startNextTopic()
            }
        }
    }

    private func move(from source: IndexSet, to destination: Int) {
        agenda.topics.move(fromOffsets: source, toOffset: destination)
    }

    private func delete(at indexSet: IndexSet) {
        agenda.topics.remove(atOffsets: indexSet)
    }
}

struct AgendaView_Previews: PreviewProvider {
    static var previews: some View {
        NavigationView {
            AgendaView()
                .environmentObject(
                    Agenda(topics: [
                        Topic(hours: 0, minutes: 30, title: "TOP 1", startDate: .now, endDate: .now.addingTimeInterval(120)),
                        Topic(hours: 0, minutes: 30, title: "TOP 2", startDate: .now.addingTimeInterval(120)),
                        Topic(hours: 2, minutes: 30, title: "TOP 3", startDate: .now.addingTimeInterval(120)),
                    ])
                )
        }
    }
}
