//
// Copyright (c) 2023 by Simon Hartmann, Mark Zimmermann
//

import CoreLocation
import MapKit
import SwiftUI
import WeatherKit

struct CitySelectionView: View {
    @StateObject var model: CitySelectionViewModel = .init()
    @State var navigationPath: NavigationPath = .init()

    var body: some View {
        NavigationStack(path: $navigationPath) {
            VStack {
                Text("Wo soll's hingehen?")
                    .font(.largeTitle.bold())

                HStack {
                    cityPreview(location: $model.firstLocation, city: model.firstCity)
                    Text("vs").font(.largeTitle.bold())
                    cityPreview(location: $model.secondLocation, city: model.secondCity)
                }

                Text("Was vergleichen?")
                    .font(.largeTitle.bold())
                    .padding(.top, 30)

                selectionButton(label: "Aktuelles Wetter") {
                    Task {
                        // Fehler sollten im produktiven Einsatz behandelt werden
                        if let weather = try? await model.compareCurrentWeather() {
                            navigationPath.append(weather)
                        }
                    }
                }

                selectionButton(label: "24 Stunden") {
                    Task {
                        // Fehler sollten im produktiven Einsatz behandelt werden
                        if let weather = try? await model.compareDayWeather() {
                            navigationPath.append(weather)
                        }
                    }
                }

                selectionButton(label: "7 Tage") {
                    Task {
                        // Fehler sollten im produktiven Einsatz behandelt werden
                        if let weather = try? await model.compareOneWeekWeather() {
                            navigationPath.append(weather)
                        }
                    }
                }

                Spacer()

                attributionMark()
            }
            .navigationDestination(for: [CityWithWeater<NowWeather>].self) { citiesWithWeather in
                CurrentWeatherComparisonView(citiesWithWeather: citiesWithWeather)
            }
            .navigationDestination(for: [CityWithWeater<HourForecast>].self) { citiesWithWeather in
                HourForecastComparisonView(citiesWithWeather: citiesWithWeather)
            }
            .navigationDestination(for: [CityWithWeater<DayForecast>].self) { citiesWithWeather in
                DayForecastComparisonView(citiesWithWeather: citiesWithWeather)
            }
            .padding()
            .background(Color(white: 0.97))
        }
    }

    private func cityPreview(location: Binding<CLLocationCoordinate2D>, city: String) -> some View {
        VStack {
            NavigationLink {
                LocationSelectionMapView(selectedCoordinate: location)
            } label: {
                LocationSelectionMapView(selectedCoordinate: location)
                    .frame(height: 150)
                    .cornerRadius(15)
                    .disabled(true)
            }

            Text(city)
                .font(.title2)
        }
    }

    private func selectionButton(label: String, action: @escaping () -> Void) -> some View {
        Button(
            action: action,
            label: {
                Text(label)
                    .font(.title2.bold())
                    .foregroundColor(.white)
            }
        )
        .frame(width: 280, height: 60)
        .background(.cyan)
        .cornerRadius(15)
    }

    private func attributionMark() -> some View {
        HStack {
            Spacer()
            AsyncImage(url: model.attributionUrl) { image in
                image.resizable().aspectRatio(contentMode: .fit).frame(width: 100)
            } placeholder: {
                ProgressView()
            }
        }
    }
}

struct CitySelectionView_Previews: PreviewProvider {
    static var previews: some View {
        CitySelectionView()
    }
}
