//
// Copyright (c) 2023 by Simon Hartmann, Mark Zimmermann
//

import Combine
import CoreLocation
import Foundation

@MainActor
class CitySelectionViewModel: ObservableObject {
    @Published var firstLocation: CLLocationCoordinate2D
    @Published var firstCity: String = ""

    @Published var secondLocation: CLLocationCoordinate2D
    @Published var secondCity: String = ""

    @Published var attributionUrl: URL?

    private var assistent: WeatherAssistant = .init()
    private var subscriptions: Set<AnyCancellable> = .init()

    init() {
        firstLocation = CLLocationCoordinate2D(latitude: 48.1548894, longitude: 11.471625)   // München
        secondLocation = CLLocationCoordinate2D(latitude: 52.506761, longitude: 13.2843081)  // Berlin

        assistent.$combinedMarkDarkURL.assign(to: &$attributionUrl)

        $firstLocation
            .sink { location in
                Task {
                    self.firstCity = try await self.cityName(at: location)
                }
            }
            .store(in: &subscriptions)

        $secondLocation
            .sink { location in
                Task {
                    self.secondCity = try await self.cityName(at: location)
                }
            }
            .store(in: &subscriptions)
    }

    func compareCurrentWeather() async throws -> [CityWithWeater<NowWeather>] {
        let firstWeather = try await assistent.getCurrentWeather(location: firstLocation)
        let secondWeather = try await assistent.getCurrentWeather(location: secondLocation)

        let weatherAtFirstLocation = CityWithWeater(city: firstCity, weather: [NowWeather(from: firstWeather)])
        let weatherAtSecondLocation = CityWithWeater<NowWeather>(city: secondCity, weather: [NowWeather(from: secondWeather)])

        return [weatherAtFirstLocation, weatherAtSecondLocation]
    }

    func compareDayWeather() async throws -> [CityWithWeater<HourForecast>] {
        let firstWeather = try await assistent.getHourWeather(location: firstLocation)
        let secondWeather = try await assistent.getHourWeather(location: secondLocation)

        let weatherAtFirstLocation = CityWithWeater(city: firstCity, weather: firstWeather.forecast.map(HourForecast.init(from:)))
        let weatherAtSecondLocation = CityWithWeater(city: secondCity, weather: secondWeather.forecast.map(HourForecast.init(from:)))

        return [weatherAtFirstLocation, weatherAtSecondLocation]
    }

    func compareOneWeekWeather() async throws -> [CityWithWeater<DayForecast>] {
        let firstWeather = try await assistent.getOneWeekForecast(location: firstLocation)
        let secondWeather = try await assistent.getOneWeekForecast(location: secondLocation)

        let weatherAtFirstLocation = CityWithWeater(city: firstCity, weather: firstWeather.map(DayForecast.init(from:)))
        let weatherAtSecondLocation = CityWithWeater(city: secondCity, weather: secondWeather.map(DayForecast.init(from:)))

        return [weatherAtFirstLocation, weatherAtSecondLocation]
    }

    private func cityName(at location: CLLocationCoordinate2D) async throws -> String {
        try await CLGeocoder().reverseGeocodeLocation(CLLocation(location), preferredLocale: .current).first?.locality ?? "unbekannter Ort"
    }
}
