//
// Copyright (c) 2023 by Simon Hartmann, Mark Zimmermann
//

import MapKit
import SwiftUI

struct LocationSelectionMapView: UIViewRepresentable {
    @Binding var selectedCoordinate: CLLocationCoordinate2D

    func makeUIView(context: Context) -> MKMapView {
        let mapView = MKMapView()
        mapView.delegate = context.coordinator
        
        let gesture = UITapGestureRecognizer(target: context.coordinator, action: #selector(Coordinator.handleTap(gestureRecognizer:)))
        mapView.addGestureRecognizer(gesture)

        context.coordinator.mapView = mapView

        return mapView
    }

    func updateUIView(_ uiView: MKMapView, context _: Context) {
        if !selectedCoordinate.latitude.isNaN && !selectedCoordinate.longitude.isNaN {
            let annotation = MKPointAnnotation()
            annotation.coordinate = selectedCoordinate
            uiView.removeAnnotations(uiView.annotations)
            uiView.addAnnotation(annotation)
        }

        uiView.setCenter(selectedCoordinate, animated: true)
    }

    func makeCoordinator() -> Coordinator {
        Coordinator(self)
    }

    class Coordinator: NSObject, MKMapViewDelegate {
        var parent: LocationSelectionMapView
        var mapView: MKMapView?

        init(_ parent: LocationSelectionMapView) {
            self.parent = parent
        }

        @objc func handleTap(gestureRecognizer: UITapGestureRecognizer) {
            let location = gestureRecognizer.location(in: gestureRecognizer.view)
            if let mapView {
                let coordinate = mapView.convert(location, toCoordinateFrom: gestureRecognizer.view)
                parent.selectedCoordinate = coordinate
            }
        }
    }
}
