//
// Copyright (c) 2023 by Simon Hartmann, Mark Zimmermann
//

import Charts
import SwiftUI
import WeatherKit

struct DayForecastComparisonView: View {
    @State private var markType: ChartMarkType = .bar

    let citiesWithWeather: [CityWithWeater<DayForecast>]

    var body: some View {
        VStack {
            CitiesView(firstCity: citiesWithWeather.first!.city, secondCity: citiesWithWeather.last!.city)

            Picker("Chart Art", selection: $markType) {
                ForEach(ChartMarkType.allCases, id: \.self) { type in
                    Text(type.name)
                }
            }
            .pickerStyle(SegmentedPickerStyle())

            ScrollView {
                temperatureChart
                rainfallAmountChart
                averageTemperatureChart
            }
        }
        .padding()
        .background(Color(white: 0.95))
    }

    var temperatureChart: some View {
        VStack {
            Text("Temperaturvergleich")
                .font(.headline)

            Chart(citiesWithWeather, id: \.city) { city in
                ForEach(city.weather, id: \.date) { day in
                    switch markType {
                    case .bar:
                        BarMark(
                            x: .value("Tag", day.date, unit: .day),
                            yStart: .value("Temperatur Min", PlottableTemperatureMeasurement(of: day.lowTemperature)),
                            yEnd: .value("Temperatur Max", PlottableTemperatureMeasurement(of: day.highTemperature)),
                            width: 10
                        )
                        .opacity(0.7)

                    case .area:
                        AreaMark(
                            x: .value("Tag", day.date, unit: .day),
                            yStart: .value("Temperatur Min", PlottableTemperatureMeasurement(of: day.lowTemperature)),
                            yEnd: .value("Temperatur Max", PlottableTemperatureMeasurement(of: day.highTemperature))
                        )
                        .interpolationMethod(.catmullRom)
                    }
                }
                .foregroundStyle(by: .value("Stadt", city.city))
            }
            .chartXAxis {
                AxisMarks(values: .stride(by: .day))
            }
            .chartXAxisLabel("Tag", alignment: .center)
            .chartYAxisLabel("°C", alignment: .center)
            .animation(.easeInOut, value: markType)
            .padding()
            .background(.white)
            .cornerRadius(5)
            .chartXAxis {
                AxisMarks(values: .automatic(minimumStride: 1)) { _ in
                    AxisGridLine()
                    AxisValueLabel(
                        format: .dateTime.week(.weekOfMonth)
                    )
                }
            }
        }
        .padding()
    }

    var rainfallAmountChart: some View {
        VStack {
            Text("Niederschlag")
                .font(.headline)

            Chart(citiesWithWeather, id: \.city) { city in
                ForEach(city.weather, id: \.date) { day in
                    switch markType {
                    case .bar:
                        BarMark(
                            x: .value("Tag", day.date, unit: .day),
                            y: .value("Niederschlag", PlottableLengthMeasurement(of: day.rainfallAmount))
                        )
                        .position(by: .value("Stadt", city.city), span: 30)

                    case .area:
                        AreaMark(
                            x: .value("Tag", day.date, unit: .day),
                            y: .value("Niederschlag", PlottableLengthMeasurement(of: day.rainfallAmount))
                        )
                        .interpolationMethod(.catmullRom)
                    }
                }
                .foregroundStyle(by: .value("Stadt", city.city))
            }
            .chartXAxis {
                AxisMarks(values: .stride(by: .day))
            }
            .chartXAxisLabel("Tag", alignment: .center)
            .chartYAxisLabel("mm", alignment: .center)
            .animation(.easeInOut, value: markType)
            .padding()
            .background(.white)
            .cornerRadius(5)
        }
        .padding()
    }

    var averageTemperatureChart: some View {
        VStack {
            Text("Durchschnittstemperatur")
                .font(.headline)

            Chart(citiesWithWeather, id: \.city) { city in
                ForEach(city.weather, id: \.date) { day in
                    LineMark(
                        x: .value("Tag", day.date, unit: .day),
                        y: .value("Durchschnitt", PlottableTemperatureMeasurement(of: day.averageTemperature))
                    )

                    PointMark(
                        x: .value("Tag", day.date, unit: .day),
                        y: .value("Durchschnitt", PlottableTemperatureMeasurement(of: day.averageTemperature))
                    )
                }
                .foregroundStyle(by: .value("Stadt", city.city))
                .symbol(by: .value("Stadt", city.city))
            }
            .chartXAxis {
                AxisMarks(values: .stride(by: .day))
            }
            .chartXAxisLabel("Tag", alignment: .center)
            .chartYAxisLabel("°C", alignment: .center)
            .animation(.easeInOut, value: markType)
            .padding()
            .background(.white)
            .cornerRadius(5)
        }
        .padding()
    }
}

struct DayForecastComparisonView_Previews: PreviewProvider {
    static var previews: some View {
        DayForecastComparisonView(
            citiesWithWeather: [
                CityWithWeater(city: "Berlin", weather: [
                    .forPreview(),
                    .forPreview(date: .now.addingTimeInterval(86400)),
                    .forPreview(date: .now.addingTimeInterval(86400 * 2)),
                ]),
                CityWithWeater(city: "München", weather: [
                    .forPreview(),
                    .forPreview(date: .now.addingTimeInterval(86400)),
                    .forPreview(date: .now.addingTimeInterval(86400 * 2)),
                ]),
            ])
    }
}
