//
// Copyright (c) 2023 by Simon Hartmann, Mark Zimmermann
//

import Charts
import SwiftUI
import WeatherKit

struct HourForecastComparisonView: View {
    @State private var markType: ChartMarkType = .bar

    let citiesWithWeather: [CityWithWeater<HourForecast>]

    var body: some View {
        VStack {
            CitiesView(firstCity: citiesWithWeather.first!.city, secondCity: citiesWithWeather.last!.city)

            Picker("Chart Art", selection: $markType) {
                ForEach(ChartMarkType.allCases, id: \.self) { type in
                    Text(type.name)
                }
            }
            .pickerStyle(SegmentedPickerStyle())

            ScrollView {
                temperatureChart
                apperantTemperatureChart
            }
        }
        .padding()
        .background(Color(white: 0.95))
    }

    var temperatureChart: some View {
        VStack {
            Text("Temperaturvergleich")
                .font(.headline)

            Chart(citiesWithWeather, id: \.city) { city in
                ForEach(city.weather, id: \.date) { hour in
                    switch markType {
                    case .bar:
                        BarMark(
                            x: .value("Stunde", hour.date),
                            y: .value("Temperatur Min", PlottableTemperatureMeasurement(of: hour.temperature)),
                            width: 10
                        )
                        .opacity(0.7)

                    case .area:
                        AreaMark(
                            x: .value("Uhrzeit", hour.date),
                            y: .value("Temperatur Min", PlottableTemperatureMeasurement(of: hour.temperature))
                        )
                        .interpolationMethod(.catmullRom)
                    }
                }
                .foregroundStyle(by: .value("Stadt", city.city))
            }
            .animation(.easeInOut, value: markType)
            .padding()
            .background(.white)
            .cornerRadius(15)
            .chartXAxisLabel("Uhrzeit", alignment: .center)
            .chartYAxisLabel("°C", alignment: .center)
            .chartXAxis {
                AxisMarks(values: .stride(by: .hour)) { value in
                    if let date = value.as(Date.self) {
                        let hour = Calendar.current.component(.hour, from: date)
                        if hour % 6 == 0 {
                            AxisGridLine().foregroundStyle(.black)
                            AxisTick().foregroundStyle(.green)
                            AxisValueLabel {
                                Text("\(hour) Uhr")
                            }
                        }
                    } else {
                        AxisGridLine().foregroundStyle(Color(white: 0.7))
                    }
                }
            }
        }
        .padding()
    }

    var apperantTemperatureChart: some View {
        VStack {
            Text("Gefühlte Temperatur")
                .font(.headline)

            Chart(citiesWithWeather, id: \.city) { city in
                ForEach(city.weather, id: \.date) { hour in
                    switch markType {
                    case .bar:
                        BarMark(
                            x: .value("Uhrzeit", hour.date, unit: .hour),
                            y: .value("Gefühlte Temperatur", PlottableTemperatureMeasurement(of: hour.apperantTemperature)),
                            width: 5
                        )
                        .position(by: .value("Stadt", city.city), span: 10)

                    case .area:
                        AreaMark(
                            x: .value("Uhrzeit", hour.date, unit: .hour),
                            y: .value("Gefühlte Temperatur", PlottableTemperatureMeasurement(of: hour.apperantTemperature))
                        )
                        .interpolationMethod(.catmullRom)
                    }
                }
                .foregroundStyle(by: .value("Stadt", city.city))
            }
            .animation(.easeInOut, value: markType)
            .padding()
            .background(.white)
            .cornerRadius(15)
            .chartXAxisLabel("Uhrzeit", alignment: .center)
            .chartYAxisLabel("°C", alignment: .center)
            .chartXAxis {
                AxisMarks(values: .stride(by: .hour)) { value in
                    if let date = value.as(Date.self) {
                        let hour = Calendar.current.component(.hour, from: date)
                        if hour % 6 == 0 {
                            AxisGridLine().foregroundStyle(.black)
                            AxisTick().foregroundStyle(.green)
                            AxisValueLabel {
                                Text("\(hour)")
                            }
                        }
                    } else {
                        AxisGridLine().foregroundStyle(Color(white: 0.7))
                    }
                }
            }
        }
        .padding()
    }
}

struct HourForecastComparisonView_Previews: PreviewProvider {
    static var previews: some View {
        HourForecastComparisonView(
            citiesWithWeather: [
                CityWithWeater(city: "Berlin", weather: [
                    .forPreview(),
                    .forPreview(date: .now.addingTimeInterval(86400)),
                ]),
                CityWithWeater(city: "München", weather: [
                    .forPreview(),
                    .forPreview(date: .now.addingTimeInterval(86400)),
                ]),
            ])
    }
}
