//
//  ApiDataStore.swift
//  NavigationSplitView
//
//  Created by Stefan Mayer-Popp on 13.05.23.
//

import Foundation

class APIDataStore: ObservableObject {
  
  @Published var breeds: Breeds? {
    didSet {
      coats = Array(
        Set( // Sicherstellen das Einträge Unique sind
          breeds?.data
            .map { $0.coat } // Extrahiere nur coats
            .filter{!$0.isEmpty} ?? [] // Filter leere Einträge
           )
      )
      .sorted()
    }
  }
  @Published var coats: [String]?

  init() {
    // Fetch breeds and publish
    fetchData(URLString: "https://catfact.ninja/breeds?limit=80") { breeds in
      DispatchQueue.main.async {
        self.breeds = breeds
      }
    }
  }
  
  // MARK: Data selection
  func breed(id: String) -> Breed? {
    breeds?.data.first(where: {$0.id == id})
  }
  
  //
  func breedsBy(coatID: Int) -> [Breed]? {
    guard let coat = coats?.first(where: {$0.id == coatID}) else { return nil }
    return breeds?.data.filter { $0.coat == coat }
  }
  
  
  
  // MARK: API Helper
  
  /// Vereinfachte Methode zum abholen von Daten eines (hoffentlich) API Endpoints mit JSON Decodable Daten :)
  /// Error Handling wird hierbei ignoriert
  /// - Parameters:
  ///   - URLString: URL Endpunkt mit JSON Daten
  ///   - completion: Ergebnis, ggf. nil falls keine Daten kommen
  func fetchData<T: Decodable>(URLString: String, completion: @escaping (T?) -> Void) {
    guard let url = URL(string: URLString) else {
      completion(nil)
      return
    }
    
    
    let task = URLSession.shared.dataTask(with: url) {
      data, response, error in
      
      let decoder = JSONDecoder()
      decoder.keyDecodingStrategy = .convertFromSnakeCase
      
      if let data = data {
        do {
          let result: T? = try decoder.decode(T.self, from: data) as T?
          completion(result)
        } catch {
          print(error)
        }
      }
    }
    task.resume()
  }
}
