//
//  ContentView.swift
//  HeiseVision
//
//  Created by Gero Gerber on 24.07.23.
//

import SwiftUI
import RealityKit
import RealityKitContent

struct ContentView: View {
    @Binding var immersionStyle: ImmersionStyle
    
    @Environment(\.openWindow) private var openWindow
    @Environment(\.dismissWindow) private var dismissWindow
    @Environment(\.openImmersiveSpace) private var openImmersiveSpace
    @Environment(\.dismissImmersiveSpace) private var dismissImmersiveSpace
    
    @State private var pickerImmersionValue: Int = 0
    
    private static let size: CGFloat = 400
    
    var body: some View {
        ScrollView {
            Grid {
                GridRow {
                    windowEntry(title: "Volume", windowId: "VolumeView")
                    windowEntry(title: "Simple Model", windowId: "SimpleModel")
                }
                GridRow {
                    windowEntry(title: "RealityView", windowId: "SimpleRealityView")
                    windowEntry(title: "UI Depth", windowId: "UIDepth")
                }
                GridRow {
                    windowEntry(title: "Ornaments", windowId: "Ornaments")
                    immersiveSpaceEntry(title: "Full Space", windowId: "ImmersiveView")
                }
                GridRow {
                    immersiveSpaceEntry(title: "ARKit", windowId: "SampleARView")
                    windowEntry(title: "Assets", windowId: "AssetsView")
                }
            }
        }
        .padding()
        .frame(maxWidth: .infinity, maxHeight: .infinity)
    }
    
    private func windowEntry(title: String, windowId: String) -> some View {
        baseEntry(title: title, windowId: windowId) {
            openWindow(id: windowId)
        } closeAction: {
            dismissWindow(id: windowId)
        } content: {
            EmptyView()
        }
    }
    
    private func immersiveSpaceEntry(title: String, windowId: String) -> some View {
        VStack {
            baseEntry(title: title, windowId: windowId) {
                Task {
                    let result = await openImmersiveSpace(id: windowId)
                    print(result)
                }
            } closeAction: {
                Task {
                    await dismissImmersiveSpace()
                }
            } content: {
                Picker("Immersion Style", selection: $pickerImmersionValue) {
                    Text("Mixed").tag(0)
                    Text("Progressive").tag(1)
                    Text("Full").tag(2)
                }
                .pickerStyle(.segmented)
                .onChange(of: pickerImmersionValue) {
                    switch pickerImmersionValue {
                    case 0: immersionStyle = .mixed
                    case 1: immersionStyle = .progressive
                    case 2: immersionStyle = .full
                    default: break
                    }
                }
                .frame(width: Self.size)
            }
        }
    }
    
    private func baseEntry(title: String, windowId: String, openAction: @escaping () -> Void, closeAction: @escaping () -> Void) -> some View {
        baseEntry(title: title, windowId: windowId, openAction: openAction, closeAction: closeAction) {
            EmptyView()
        }
    }
    
    private func baseEntry<Content: View>(title: String, windowId: String, openAction: @escaping () -> Void, closeAction: @escaping () -> Void, @ViewBuilder content: @escaping () -> Content) -> some View {
        VStack {
            Text(title)
                .font(.title)
            HStack {
                Button("Open") {
                    openAction()
                }
                Button("Close") {
                    closeAction()
                }
            }
            Spacer()
            content()
            Spacer()
        }
        .frame(minWidth: Self.size, minHeight: 200)
        .padding()
        .background(.regularMaterial, in: .rect(cornerRadius: 12))
        .frame(maxWidth: .infinity, maxHeight: .infinity)
    }
}

#Preview {
    ContentView(immersionStyle: .constant(.mixed))
}
