//
//  Heise Swift Macros
//  Copyright © 2023 Gero Gerber. All rights reserved.
//

import SwiftCompilerPlugin
import SwiftDiagnostics
import SwiftSyntax
import SwiftSyntaxBuilder
import SwiftSyntaxMacros

public struct TrackChangeMacro: ExpressionMacro {
    public static func expansion(
        of node: some FreestandingMacroExpansionSyntax,
        in context: some MacroExpansionContext
    ) -> ExprSyntax {
        guard let argument = node.argumentList.first?.expression else {
            fatalError("Compiler bug: the macro does not have any arguments")
        }

        guard let tupleArgument = argument.as(TupleExprSyntax.self) else {
            fatalError("Compiler bug: argument must be a tuple")
        }

        if tupleArgument.elements.count != 2 {
            fatalError("Compiler bug: Tuple must only contain two elements")
        }

        let closureElements = tupleArgument.elements.compactMap { $0.expression.as(ClosureExprSyntax.self) }

        if closureElements.count != 2 {
            context.diagnose(Diagnostic(node: node, message: TrackChangeDiagnostic.parameterNotATuple))
            return "{}()"
        }

        let uniqueName = context.makeUniqueName("trackChange")
        let signature0: SyntaxProtocol = closureElements[0].signature ?? TokenSyntax("")
        let signature1: SyntaxProtocol = closureElements[1].signature ?? TokenSyntax("")

        return
            """
            {
                \(uniqueName)()
                @Sendable func \(uniqueName)() {
                    withObservationTracking { \(signature0)
                        \(closureElements[0].statements)
                    } onChange: { \(signature1)
                        \(closureElements[1].statements)
                        \(uniqueName)()
                    }
                }
            }()
            """
    }
}

@main
struct TrackChangePlugin: CompilerPlugin {
    let providingMacros: [Macro.Type] = [
        TrackChangeMacro.self
    ]
}
