# training.py

# Train your machine learning and AI models on Apple GPUs
# https://developer.apple.com/videos/play/wwdc2024/10160/

# Tipps:
# 1. Python 3.10 instalieren
# brew install python@3.10
# 
# 2. Virtuelle Umgebung für Python einrichten...
# python3.10 -m venv .venv
# ...und aktivieren
# source .venv/bin/activate
# 
# 3. Voraussetzungen installieren
# pip3 install torch transformers datasets peft "huggingface_hub[cli]"
# pip3 install coremltools numpy==1.24.1
#
# 4. Konto auf Hugging Face (HF) einrichten
# 
# 5. Token für lesenden Zugriff generieren und kopieren
# 
# 6. Auf HF anmelden und kopiertes Token als Passwort verwenden
# huggingface-cli
#

###
# PyTorch Modell von HF laden
###
import torch

from transformers import GPT2LMHeadModel, GPT2Tokenizer

# Abfrage des PyTorch Modells mit Textein- und -ausgabe
def respond(prompt, model, tokenizer):
    input_ids = tokenizer.encode(prompt, return_tensors='pt')
    attention_mask = torch.ones(input_ids.shape, dtype=torch.long)
    with torch.no_grad():
        model.eval() # Abfragemodus aktivieren
        output = model.generate(input_ids, attention_mask=attention_mask, max_length=50, num_return_sequences=1, pad_token_id=tokenizer.eos_token_id)
    response = tokenizer.decode(output[0], skip_special_tokens=True)
    return response

# PyTorch Modell im Dateisystem speichern
def save_model(path, model, tokenizer=None):
    if not tokenizer == None:
        tokenizer.save_pretrained(path)
    model.save_pretrained(path)

# PyTorch Modell aus Dateisystem oder von HF laden
def load_model(path):
    model = GPT2LMHeadModel.from_pretrained(path)
    tokenizer = GPT2Tokenizer.from_pretrained(path)
    return model, tokenizer

# Vortrainiertes GPT-2 Modell und Tokenizer von HF laden
model_name = 'gpt2'
model = GPT2LMHeadModel.from_pretrained(model_name, torchscript=True)
tokenizer = GPT2Tokenizer.from_pretrained(model_name, clean_up_tokenization_spaces=True)

# Abfrage (Inferenz) durchführen
prompt = "Would you like to travel around the world?"
response = respond(prompt, model, tokenizer)
print(response)

# Modell und Tokenizer im Dateisystem speichern
model_path = './gpt2_model'
save_model(model_path, model, tokenizer)

# Modell aus Dateisystem laden
saved_model, saved_tokenizer = load_model(model_path)

# Inferenz mit Modell aus Dateisystem durchführen
response = respond(prompt, saved_model, saved_tokenizer)
print(response)

###
# PyTorch Modell in Core ML konvertieren
###
import coremltools as ct
import numpy as np

# Abfrage des Core ML Modells mit Textein- und -ausgabe
def mlrespond(prompt, model):
    input_tnsr = torch.tensor(tokenizer.encode(prompt))
    input_dict = {"input": input_tnsr.to(torch.int32).numpy()}
    output = model.predict(input_dict)
    
    # hack: inspect dict to look for `logits´
    # for key, value in output.items():
    #     print(f"Key: {key}, Shape: {value.shape}, Samples: {value.flatten()[:]}")
    
    # convert logits to probabilities
    probs = torch.softmax(torch.tensor(output["linear_0"]), dim=-1)
    # probs = torch.softmax(torch.tensor(output["linear_24"]), dim=-1)
    # convert probabilities to tokens
    tokns = torch.argmax(probs, dim=-1).tolist()
    response = tokenizer.decode(tokns)
    return response

# PyTorch Modell in Core ML konvertieren
def convert(model):
    rnd_input = torch.randint(50527, (5,))
    trc_model = torch.jit.trace(model, rnd_input)
    scr_model = torch.jit.script(trc_model)
    mlmodel = ct.convert(scr_model, inputs=[ct.TensorType(name="input", shape=(ct.RangeDim(1, 1024),), dtype=np.int32)])
    return mlmodel

# Modell aus Dateisystem in Core ML konvertieren
mlmodel = convert(saved_model)

# Core ML Modell im Dateisystem speichern
mlmodel_path = model_path + ".mlpackage"
mlmodel.save(mlmodel_path)

# Core ML Modell aus Dateisystem laden
saved_mlmodel = ct.models.MLModel(mlmodel_path)

# Inferenz mit Core ML Modell aus Dateisystem durchführen
response = mlrespond(prompt, saved_mlmodel)
print(response)

###
# PyTorch Modell nachtrainieren (Finetuning)
###

# Adapterkonfiguration einstellen...
from peft import LoraConfig, TaskType
lora_config = LoraConfig(task_type=TaskType.CAUSAL_LM, inference_mode=False, r=8, lora_alpha=32, lora_dropout=0.1)

# ...und Adapter erzeugen
from peft import PeftModel
adapter = PeftModel(model, lora_config)

# Datensatz mit Texten von Shakespeare laden
from transformers import TextDataset
shakespeare = TextDataset(tokenizer=tokenizer, file_path="tiny_shakespeare.txt", block_size=128)[:256]

# Trainerkonfiguration einstellen...
from transformers import DataCollatorForLanguageModeling
data_collator = DataCollatorForLanguageModeling(tokenizer=tokenizer, mlm=False)

from transformers import TrainingArguments, Trainer
adapter_path = model_path + "_shakespeare"
training_args = TrainingArguments(output_dir=adapter_path + "_lora", num_train_epochs=5, per_device_train_batch_size=2)

# ...Trainer erzeugen...
trainer = Trainer(model=adapter, args=training_args, data_collator=data_collator, train_dataset=shakespeare)

# ...und Modell nachtrainieren
trainer.train()

# Inferenz mit nachtrainietem Modell durchführen
response = respond(prompt, adapter)
print(response)

# Shakespeare Adapter im Dateisystem speichern
save_model(adapter_path, ftmodel, tokenizer)

# Shakespeare Adapter aus Dateisystem ins Model laden
model.load_adapter(adapter_path)

# Inferenz mit nachtrainietem Modell aus Dateissystem durchführen
response = respond(prompt, model, tokenizer)
print(response)

# In Core ML Modell konvertieren
mlmodel = convert(model)

# Als Core ML Modell im Dateisystem speichern
mlmodel_path = adapter_path + ".mlpackage"
mlmodel.save(mlmodel_path)
