//
//  ContentView.swift
//  HeiseFoundationModels
//
//  Created by Gero Gerber on 08.09.25.
//

import FoundationModels
import SwiftUI

fileprivate struct GetJokeTool: Tool {
    let name = "getJoke"
    let description = "Retrieve jokes by a keyword"
    
    @Generable
    struct Arguments {
        @Guide(description: "Keyword to search for jokes")
        var keyword: String
    }
    
    struct JokeResponse: Decodable {
        let joke: String?
    }
    
    enum GetJokeToolError: Error {
        case noJoke
    }
    
    func call(arguments: Arguments) async throws -> String {
        let url = URL(string: "http://v2.jokeapi.dev/joke/Any?type=single")!
        let (data, _) = try await URLSession.shared.data(from: url)
        let jokeResponse = try JSONDecoder().decode(JokeResponse.self, from: data)
        if let joke = jokeResponse.joke {
            return joke
        } else {
            return "No joke for you!"
        }
    }
}

struct ContentView: View {
    @State private var session = LanguageModelSession(tools: [GetJokeTool()])
    @State private var response: String?
    
    var body: some View {
        VStack {
            ScrollView {
                Text(response ?? "")
            }
            Button("Generate Text") {
                Task {
                    do {
                        let result = try await session.respond(to: "Generate random text and embed a joke within")
                        
                        await MainActor.run {
                            response = result.content
                        }
                    } catch {
                        print(error)
                    }
                }
            }
            .buttonStyle(BorderedProminentButtonStyle())
            .disabled(session.isResponding)
        }
        .padding()
        .onAppear {
            session.prewarm()
        }
    }
}

#Preview {
    ContentView()
}
