#!/usr/bin/perl

use File::Copy;
use Git::Repository;
use strict;
use warnings;

$| = 1;
#################################################################
# Move files from cisco archive upload-directory into git-repo
# 22/10/2015 H.Kessener
#################################################################
# Config section

my $DEBUG = 1;

# == Patterns ==
# process files matching this pattern:
my $CHECK_PATTERN = 'gate|switch';
# split filenames using this pattern:
my $SPLIT_PATTERN = '\+';
# == Directories (no trailing slash) ==
# directory for ftp file uploads:
my $FTP_DIRECTORY = '/srv/ftp/cisco/upload';
# directory of git-repository:
my $GIT_DIRECTORY = '/srv/git/cisco';
# == Times (seconds) ==
# process ftp files older than:
my $ACCEPT_TIME = time() - 30;
# delete ftp files older than:
my $DELETE_TIME = time() - 10*24*3600;
# number of files to keep (per device)
my $FILES_TO_KEEP = 10;
# == Central repository
# push to URL/path (leave empty if unused)
my $CENTRAL_REPO = '';

#################################################################
# Get list of files to be processed

my %FileHash;

# open upload-directory
opendir(DIR,$FTP_DIRECTORY) or die;
foreach my $FileName (readdir(DIR)) {
  # skip dot-files
  next if($FileName =~ /^\./);
  # skip files not matching check-pattern
  next unless($FileName =~ /$CHECK_PATTERN/);
  # skip files not matching split-pattern
  next unless($FileName =~ /$SPLIT_PATTERN/);
  #
  ######## quick insertion for IOS bug fix begin ########
  # check if minimum file size is reached
  my $FileSize = (stat(qq|$FTP_DIRECTORY/$FileName|))[7];
  if($FileSize <= 199) {
    # this seems to be an "empty" file; remove it.
    ($DEBUG >= 2) and print(qq|--> deleting "$FileName"\n|);
    # (no debian-7-pkg for sudo)
    system(qq|/usr/bin/sudo /bin/rm $FTP_DIRECTORY/$FileName|);
    next;
  }
  ######## quick insertion for IOS bug fix end ########
  #
  # store filetime for being able to sort
  my $FileTime = (stat(qq|$FTP_DIRECTORY/$FileName|))[9];
  # filename is something like "switch-s5-2+Feb-11-15-12-20-MET-2":
  # -> we use the first part "switch-s5-2" as the hostname,
  # -> the rest may be valid or rubbish, we ignore it.
  my($HostName,$foo) = split(/$SPLIT_PATTERN/,$FileName);
  # compose a unique and sortable hash key 
  $FileHash{$HostName}->{$FileTime} = $FileName;
}
closedir(DIR);

unless(keys(%FileHash)) {
  ($DEBUG >= 2) and print(qq|No files found\n|);
  exit(1);
}

#################################################################
# Process files

# Git object
my $Git = Git::Repository->new(work_tree => $GIT_DIRECTORY) or die;

# Only push after commit
my $didCommit = 0;

# Copy most recent files into git-directory,
# remove outdated files from upload-directory
foreach(sort(keys(%FileHash))) {
  # iterate over hosts, sorted for better readability
  my $HostName = $_;
  my $HostPtr  = $FileHash{$HostName};
  ($DEBUG >= 2) and print(qq|Processing host "$_"\n|);

  my @FilesToDo = keys(%{$HostPtr});
  my @FilesDone;
  foreach(sort {$b cmp $a } (@FilesToDo)) {
    # iterate over files for this host in
    # descending order (most recent file first)
    my $FileTime = $_;
    my $FileName = $HostPtr->{$FileTime};
    ($DEBUG >= 3) and print(qq|Processing file "$FileName"\n|);

    # copy over to git-directory
    if((@FilesDone == 0)&&($FileTime <= $ACCEPT_TIME)) {
      copy(qq|$FTP_DIRECTORY/$FileName|,qq|$GIT_DIRECTORY/$HostName|)
      or warn(qq|copying to git-folder failed\n|);
      push(@FilesDone,$FileName);
      $Git->run(add => $HostName);
      my $foo = $Git->run(commit => '-m',$HostName);
      unless($foo =~ /clean/) {
        ($DEBUG >= 1) and print(qq|-> committed "$FileName"\n|);
        $didCommit++;
      }
      # that's it, we just observe the most recent version
    }

    # delete outdated files from upload-directory
    if((@FilesToDo > $FILES_TO_KEEP)&&($FileTime <= $DELETE_TIME)) {
      ($DEBUG >= 1) and print(qq|-> deleting "$FileName"\n|);
      # (no debian-7-pkg for sudo)
      system(qq|/usr/bin/sudo /bin/rm $FTP_DIRECTORY/$FileName|);
    }
  }
}

# Push to central repository (optional)
if(($didCommit > 0)&&($CENTRAL_REPO ne '')) {
  ($DEBUG >= 2) and print(qq|Pushing to central repo\n|);
  chdir($GIT_DIRECTORY) or die;
  $Git->run(push => $CENTRAL_REPO,'master');
}

#################################################################
1;
