//
//  SessionContainer.swift
//  HeiseMultipeerConnectivity
//
//  Created by Gero Gerber on 19.02.19.
//  Copyright © 2019 Gero Gerber. All rights reserved.
//

import Foundation
import MultipeerConnectivity

class SessionContainer: NSObject {
    var sessions: Set<MCSession> {
        get {
            return Set<MCSession>(activeSessions)
        }
    }
    
    weak var delegate: SessionsDelegate?
    
    private let localPeer: LocalPeer
    private let chatHistory: ChatHistory
    private var activeSessions = Set<MCSession>()
    
    init(localPeer: LocalPeer, chatHistory: ChatHistory) {
        self.localPeer = localPeer
        self.chatHistory = chatHistory
    }
    
    func createSession() -> MCSession {
        let session = MCSession(peer: localPeer.peerId)
        session.delegate = self
        activeSessions.insert(session)
        delegate?.sessionAdded(session: session)
        print("Created session")
        return session
    }
    
    func findSession(with peerId: MCPeerID) -> MCSession? {
        print("Search session with peer '\(peerId.displayName)'")
        
        for s in activeSessions {
            if s.connectedPeers.count == 1 && s.connectedPeers[0] == peerId {
                print("Found session with only peer '\(peerId.displayName)'")
                return s
            }
        }
        
        print("No existing session found with peer '\(peerId.displayName)'")
        
        return nil
    }
    
    func removeSession(session: MCSession) {
        activeSessions.remove(session)
        delegate?.sessionRemoved(session: session)
        print("Removed session")
    }
    
    private class func sessionStateDescription(_ state: MCSessionState) -> String {
        switch state {
        case .notConnected:
            return "Not Connected"
        case .connecting:
            return "Connecting"
        case .connected:
            return "Connected"
        }
    }
}

extension SessionContainer: MCSessionDelegate {
    func session(_ session: MCSession, peer peerID: MCPeerID, didChange state: MCSessionState) {
        print("Peer '\(peerID.displayName)' state changed to '\(SessionContainer.sessionStateDescription(state))'")
        DispatchQueue.main.async { [weak self] in
            if let strongSelf = self {
                strongSelf.delegate?.peerStateChange(peerID: peerID, state: state)
                
                if state == .notConnected && session.connectedPeers.count == 0 {
                    strongSelf.removeSession(session: session)
                }
            }
        }
    }
    
    func session(_ session: MCSession, didReceive data: Data, fromPeer peerID: MCPeerID) {
        print("Received data from Peer '\(peerID.displayName)'")
        DispatchQueue.global().async {
            do {
                if let chatMessage = try NSKeyedUnarchiver.unarchiveTopLevelObjectWithData(data) as? ChatMessage {
                    DispatchQueue.main.async { [weak self] in
                        self?.chatHistory.addChatMessage(session: session, message: chatMessage)
                    }
                }
            } catch {
                print("Failed to unarchive Chat-Message: \(error.localizedDescription)")
            }
        }
    }
    
    func session(_ session: MCSession, didReceive stream: InputStream, withName streamName: String, fromPeer peerID: MCPeerID) {
        print("Received stream from Peer '\(peerID.displayName)'")
    }
    
    func session(_ session: MCSession, didStartReceivingResourceWithName resourceName: String, fromPeer peerID: MCPeerID, with progress: Progress) {
        print("Received start receiving recource '\(resourceName)' from Peer '\(peerID.displayName)'")
    }
    
    func session(_ session: MCSession, didFinishReceivingResourceWithName resourceName: String, fromPeer peerID: MCPeerID, at localURL: URL?, withError error: Error?) {
        print("Received finished receiving recource '\(resourceName)' from Peer '\(peerID.displayName)'")
    }
}

extension MCSession {
    func displayName() -> String {
        if self.connectedPeers.count == 0 {
            return ""
        }
        
        var result = ""
        
        for p in self.connectedPeers {
            result.append(p.displayName + " ")
        }
        
        return result
    }
}

protocol SessionsDelegate: class {
    func sessionAdded(session: MCSession)
    func sessionRemoved(session: MCSession)
    func peerStateChange(peerID: MCPeerID, state: MCSessionState)
}
