//
//  SessionViewController.swift
//  HeiseMultipeerConnectivity
//
//  Created by Gero Gerber on 19.02.19.
//  Copyright © 2019 Gero Gerber. All rights reserved.
//

import AVFoundation
import MobileCoreServices
import MultipeerConnectivity
import UIKit

class SessionViewController: UIViewController {

    @IBOutlet var tableView: UITableView!
    @IBOutlet var textField: UITextField!
    @IBOutlet var bottomConstraint: NSLayoutConstraint!
    
    var session: MCSession?
    var multipeerConnectivityProvider: MultipeerConnectivityProvider?
    var chatMessages = [ChatMessage]()
    var scrolledDown = true
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        navigationController?.delegate = self
        
        tableView.delegate = self
        tableView.dataSource = self
        tableView.separatorStyle = .none
        tableView.rowHeight = UITableView.automaticDimension
        tableView.estimatedRowHeight = 600
        
        multipeerConnectivityProvider = AppDelegate.shared.multipeerConnectivityProvider
        
        let gesture = UITapGestureRecognizer(target: self, action: #selector(handleTapGesture))
        gesture.numberOfTapsRequired = 1
        gesture.numberOfTouchesRequired = 1
        view.addGestureRecognizer(gesture)
        
        textField.delegate = self
    }
    
    override func viewDidAppear(_ animated: Bool) {
        multipeerConnectivityProvider?.sessionsDelegate = self
        multipeerConnectivityProvider?.chatsDelegate = self
        
        if let session = session {
            title = session.displayName()
            if let previousChatMessages = multipeerConnectivityProvider?.getChatMessages(for: session) {
                chatMessages = previousChatMessages
            }
        }
    
        addKeyboardObserver()
        refresh()
    }
    
    override func viewDidDisappear(_ animated: Bool) {
        removeKeyboardObserver()
    }
    
    private func refresh() {
        tableView.reloadData()
        
        if scrolledDown {
            scrollToBottom()
        }
    }

    @IBAction private func sendTouchUpInside(_ sender: Any) {
        sendMessage()
    }
    
    @IBAction private func sendPictureTouchUpInside(_ sender: Any) {
        let vc = UIImagePickerController()
        vc.sourceType = .photoLibrary
        vc.mediaTypes = [kUTTypeImage as String, kUTTypeMovie as String]
        vc.allowsEditing = true
        vc.delegate = self
        present(vc, animated: true)
    }
    
    @IBAction private func takePictureTouchUpInside(_ sender: Any) {
        let vc = UIImagePickerController()
        vc.sourceType = .camera
        vc.allowsEditing = true
        vc.delegate = self
        present(vc, animated: true)
    }
    
    private func sendMessage() {
        if let session = session {
            if let text = textField.text, text.count > 0 {
                multipeerConnectivityProvider?.sendTextMessage(to: session, with: text)
                textField.text? = ""
                dismissKeyboard()
            }
        }
    }
    
    private func scrollToBottom() {
        let numberOfRows = tableView.numberOfRows(inSection: 0)
        
        if numberOfRows > 0 {
            let indexPath = NSIndexPath(item: numberOfRows - 1, section: 0)
            tableView.scrollToRow(at: indexPath as IndexPath, at: UITableView.ScrollPosition.bottom, animated: false)
        }
    }
}

// MARK: - UIViewController

extension SessionViewController {
    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        if let destination = segue.destination as? AddPeerTableViewController {
            destination.session = session
        } else if let destination = segue.destination as? GiphyTableViewController {
            destination.delegate = self
        }
    }
}

// MARK: - SessionsDelegate

extension SessionViewController: SessionsDelegate {
    func sessionAdded(session: MCSession) {
        
    }
    
    func sessionRemoved(session: MCSession) {
        
    }
    
    func peerStateChange(peerID: MCPeerID, state: MCSessionState) {
        if let session = session {
            if session.connectedPeers.count > 0 {
                title = session.displayName()
            } else {
                title = "No Peers"
            }
        } else {
            title = "No Session"
        }
    }
}

// MARK: - ChatHistoryDelegate

extension SessionViewController: ChatHistoryDelegate {
    func chatMessageAdded(session: MCSession, message: ChatMessage) {
        if session == self.session {
            chatMessages.append(message)
            tableView.performBatchUpdates({
                let indexPath = IndexPath(row: chatMessages.count - 1, section: 0)
                tableView.insertRows(at: [indexPath], with: .bottom)
            }, completion: { [weak self] _ in
                self?.scrollToBottom()
            })
        }
    }
}

// MARK: - UITableViewDataSource

extension SessionViewController: UITableViewDataSource {
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return chatMessages.count
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let chatMessage = chatMessages[indexPath.row]
        var cellIdentifier = ""
        
        switch chatMessage.type! {
        case .text:
            cellIdentifier = chatMessage.sentFromLocalDevice ? "TextMessageSentCell" : "TextMessageReceivedCell"
        case .image:
            cellIdentifier = chatMessage.sentFromLocalDevice ? "ImageMessageSentCell" : "ImageMessageReceivedCell"
        case .video:
            cellIdentifier = chatMessage.sentFromLocalDevice ? "VideoMessageSentCell" : "VideoMessageReceivedCell"
        }
        
        let cell = tableView.dequeueReusableCell(withIdentifier: cellIdentifier, for: indexPath)
        
        if var messageCell = cell as? MessageTableViewCellProtocol {
            messageCell.delegate = self
            messageCell.setChatMessage(message: chatMessage)
        }
        
        return cell
    }
}

// MARK: - UITextFieldDelegate

extension SessionViewController: UITextFieldDelegate {
    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        sendMessage()
        textField.resignFirstResponder()
        return false
    }
}

// MARK: - UITableViewDelegate

extension SessionViewController: UITableViewDelegate {
    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        let height = scrollView.frame.size.height
        let contentHeight = scrollView.contentSize.height
        let scrollOffset = scrollView.contentOffset.y
        
        scrolledDown = (scrollOffset + height) >= contentHeight
    }
}

// MARK: - UINavigationControllerDelegate

extension SessionViewController: UINavigationControllerDelegate {
    
}

// MARK: - UIImagePickerControllerDelegate

extension SessionViewController: UIImagePickerControllerDelegate {
    func imagePickerController(_ picker: UIImagePickerController, didFinishPickingMediaWithInfo info: [UIImagePickerController.InfoKey : Any]) {
        picker.dismiss(animated: true)
        
        if let session = session {
            if let image = info[.editedImage] as? UIImage {
                multipeerConnectivityProvider?.sendImageMessage(to: session, with: image)
            } else if let mediaType = info[.mediaType] as? String {
                if mediaType == String(kUTTypeMovie) {
                    if let mediaURL = info[.mediaURL] as? URL {
                        multipeerConnectivityProvider?.sendVideoMessage(to: session, with: mediaURL)
                    }
                }
            }
        }
    }
}

// MARK: - Keyboard Handling

extension SessionViewController {
    @objc func handleTapGesture(_ sender: UITapGestureRecognizer) {
        dismissKeyboard()
    }
    
    func dismissKeyboard() {
        view.endEditing(true)
    }
    
    func addKeyboardObserver() {
        NotificationCenter.default.addObserver(self, selector: #selector(keyboardWillChange), name: UIResponder.keyboardWillChangeFrameNotification, object: nil)
    }
    
    func removeKeyboardObserver() {
        NotificationCenter.default.removeObserver(self, name: UIResponder.keyboardWillChangeFrameNotification, object: nil)
    }
    
    @objc func keyboardWillChange(notification: Notification) {
        var screenFrameBegin = CGRect()
        var screenFrameEnd = CGRect()
        
        if let value = notification.userInfo?[UIResponder.keyboardFrameBeginUserInfoKey] as? NSValue {
            screenFrameBegin = value.cgRectValue
        }
        
        if let value = notification.userInfo?[UIResponder.keyboardFrameEndUserInfoKey] as? NSValue {
            screenFrameEnd = value.cgRectValue
        }
        
        let frameBegin = view.convert(screenFrameBegin, from: view.window)
        let frameEnd = view.convert(screenFrameEnd, from: view.window)
        let originDelta = frameEnd.origin.y - frameBegin.origin.y
        
        bottomConstraint.constant -= originDelta
        view.setNeedsUpdateConstraints()
        
        var animatedCurve = UIView.AnimationCurve.easeInOut.rawValue
        
        if let number = notification.userInfo?[UIResponder.keyboardAnimationCurveUserInfoKey] as? NSNumber {
            animatedCurve = number.intValue
        }
        
        var animationDuration = 0.25
        
        if let number = notification.userInfo?[UIResponder.keyboardAnimationDurationUserInfoKey] as? NSNumber {
            animationDuration = number.doubleValue
        }
        
        UIView.animate(withDuration: animationDuration, delay: 0.0, options: UIView.AnimationOptions(rawValue: UInt(animatedCurve) << 16), animations: { [weak self] in
            self?.view.layoutIfNeeded()
            }, completion: nil)
    }
}

// MARK: - MessageTableViewCellDelegate

extension SessionViewController: MessageTableViewCellDelegate {
    func presentViewController(vc: UIViewController) {
        present(vc, animated: true, completion: nil)
    }
}

// MARK: - GiphyTableViewControllerDelegate

extension SessionViewController: GiphyTableViewControllerDelegate {
    func giphySelected(data: Data, pathExtension: String) {
        if let session = session {
            multipeerConnectivityProvider?.sendVideoMessage(to: session, with: data, type: pathExtension, repeat: 10)
        }
    }
}
