/*
GPS Demospiel

GPS-Empfänger + LCD Anzeige.
Eingegeben wird ein Zielpunkt: lat2, lon2
Abhängig von der aktuellen Entfernung zum Zielpunkt werden unterschiedliche Daten auf dem Display ausgegeben.
Im Augenblick sind 4 Optionen aktiv, DISPSEL 1 ..4.

Um die Rechnung zu beschleunigen, berechnet das Programm Entfernungen über einen festen Breitengrad (mylat = 51.0)

Vers. 2014-08-03 rp


*/

#include <TinyGPS++.h>
#include <LiquidCrystal.h>
#include <SoftwareSerial.h>

LiquidCrystal lcd(12, 11, 7, 6, 5, 4);  //  LCD connection; 6 lines so R/W -> ground

static const int RXPin = 2, TXPin = 3;  //  GPS connection
static const uint32_t GPSBaud = 9600;

TinyGPSPlus gps;
SoftwareSerial ss(RXPin, TXPin); 

//convert degree into m
const float mylat = 51.0; // i.e living around 50 deg lattitute
const long int lon2km = 111L * cos(mylat/180.0 * 3.1415); //convert lon to km
const long int lat2km = 111L; //convert lat to km
 
const byte LON = 0;
const byte LAT = 1;

//***  Target location {lon , lat} ***
const double target[2] = {7.010111, 50.315306}; 
char targetname[] = "Booser Eifelturm";

double distold=0, distnew=0, distd; 
double locnew[2] = {0,0};

int fcourse;
char sbuf0[32]="", sbuf1[32]="";


byte DISPSEL = 1; 

void setup() {
  Serial.begin(115200);
  ss.begin(GPSBaud);
  Serial.println(F("GPS LCD test"));
  
  lcd.begin(16, 2); // 6 columns x 2 rows: 
  lcd.setCursor(0,0);
  lcd.print("Pls wait..");
  lcd.setCursor(0,1); 
  lcd.print("1-10 min");
}


void loop()
{
  while (ss.available() > 0){
    if (gps.encode(ss.read()))
      displayInfo();
   
  }
  if (millis() > 5000 && gps.charsProcessed() < 10)
  {
    Serial.println(F("Kein Eingangssignal"));
    while(true);
  }
}

  
void displayInfo()
{ 
  double lonfrac, latfrac, lonint, latint;
  char tbuflon[16]="", tbuflat[16]="", tbufdist[16]="";
  char c;
  
  if (gps.location.isValid()) {
   
    locnew[LON] =gps.location.lng();
    locnew[LAT] =gps.location.lat();

    distold= distnew;
    distnew= simpledistance(locnew[LAT], locnew[LON], target[LAT], target[LON]);
    
    distd = distold-distnew;  
    fcourse= gps.courseTo(locnew[LAT], locnew[LON], target[LAT], target[LON]);
  
    strcpy(sbuf0,"");
    strcpy(sbuf1,"");

    //  selector for distant dependent display
    DISPSEL = 1; // lon, lat: nn.nnnn; dist, course, tendency
    if (distnew < 1000)
      DISPSEL = 2;  // lon, lat: nnnnnn; dist, course tendency
    if (distnew < 300)
      DISPSEL = 3;  // hello; dist, course, tendency
    if (distnew < 50)
      DISPSEL = 4;  // height; dist, course, tendency 
      
    //DISPSEL = 4;  //test

    switch (DISPSEL){
      case 1: // display delta coordinates (not used)
      
        dtostrf((locnew[LAT]-target[LAT])*lat2km,6,2, tbuflat);
        dtostrf((locnew[LON]-target[LON])*lon2km,6,2, tbuflon);
        sprintf(sbuf0,"%s %skm", tbuflat, tbuflon); 
   
        dtostrf(distnew,6,0, tbufdist);
        sprintf(sbuf1,"%s m %3s  ", tbufdist, cardinalDE(fcourse));
        break;
      case 2: // near target, switch from km -> m
        //lcd.clear();
        dtostrf((locnew[LAT]-target[LAT])*lat2km*1000,6,0, tbuflat);
        dtostrf((locnew[LON]-target[LON])*lon2km*1000,6,0, tbuflon);
        sprintf(sbuf0,"%s %s m", tbuflat, tbuflon); 
   
        sprintf(sbuf1,"%6d m %3s ", (int) distnew, cardinalDE(fcourse));
        break;
      case 3: // very near target, reveal name
        strcpy(sbuf0, targetname);
        sprintf(sbuf1,"%d m h: %d m      ", (int) distnew, (int) gps.altitude.meters());
        break; 
      case 4: // reached, reveal task
        strcpy(sbuf0,"Wie hoch?       ");
        sprintf(sbuf1,"%d m; h: %d m      ", (int) distnew, (int) gps.altitude.meters());
        break;   
      case 11: // display coordinates (not used in this context)
        lonfrac = modf(locnew[LON], &lonint);
        lonfrac *= 10000;
        latfrac = modf(locnew[LAT], &latint);
        latfrac *= 10000;
        sprintf(sbuf0,"%2d.%4d %2d.%4d",(int) latint, (int) latfrac, (int) lonint, (int) lonfrac);
   

        sprintf(sbuf1,"%6d m %3s ", (int) distnew, cardinalDE(fcourse));
        break;
      case 12: // display fractional part of coordinates (not used in this context)
        //lcd.clear();
        lonfrac = modf(locnew[LON], &lonint);
        lonfrac *= 1000000;
        latfrac = modf(locnew[LAT], &latint);
        latfrac *= 1000000;
        dtostrf(lonfrac,6,0,tbuflon);
        dtostrf(latfrac,6,0,tbuflat);
        sprintf(sbuf0,".%s .%s", tbuflat, tbuflon);


        sprintf(sbuf1,"%6d m %3s  ", (int) distnew, cardinalDE(fcourse));
        break;
    }

    printlcd();
    printmonitor();
    
    /*
    // monitor position if PC is connected  
    Serial.println();
    Serial.print(F("Position: "));
    Serial.print(locnew[LAT],6);
    Serial.print("; ");
    Serial.println(locnew[LON],6);
    */

  }
  else
  {
    Serial.print(F("INVALID"));
  }
  

  
  // monitor time
  
  Serial.print(F(" "));
  if (gps.time.isValid() )
  {
    Serial.print(F("Time: "));
    if (gps.time.hour() < 10) Serial.print(F("0"));
    Serial.print(gps.time.hour());
    Serial.print(F(":"));
    if (gps.time.minute() < 10) Serial.print(F("0"));
    Serial.print(gps.time.minute());
    Serial.print(F(":"));
    if (gps.time.second() < 10) Serial.print(F("0"));
    Serial.print(gps.time.second());
    Serial.print(F("."));
    if (gps.time.centisecond() < 10) Serial.print(F("0"));
    Serial.print(gps.time.centisecond());
  }
  else
  {
    Serial.print(F("INVALID"));
  }

  Serial.println();
}

void printlcd(){
  //LCD output
  lcd.setCursor(0,0);
  lcd.print(sbuf0);
  lcd.setCursor(0,1);   
  lcd.print(sbuf1);  
}

void printmonitor(){
  //Monitor output
  Serial.println(F("*****************"));
  Serial.print("*");
  Serial.print(sbuf0);
  Serial.println("");
  Serial.print("*");
  Serial.print(sbuf1);
  Serial.println("");
  Serial.println(F("*****************"));
  Serial.println();
}


double simpledistance(double lata, double lona, double latb, double lonb){
  //quick distance calculator; 
  return sqrt((lata-latb)*lat2km * (lata-latb)*lat2km  + (lona-lonb)*lon2km * (lona-lonb)*lon2km)*1000;
}

void testdistance(){
  Serial.print("d1: 361'000 m - ");
  Serial.println(simpledistance( 50.123,  7.123,  53.123,  9.123)); //361 km
  Serial.print("d2: 132'000 m - ");
  Serial.println(simpledistance( 50.123,  7.123,  51.123,  8.123)); //132 km
  Serial.print("d3: 13'000 m - ");
  Serial.println(simpledistance( 50.123,  7.123,  50.223, 7.229)); //13 km
  Serial.print("d4: 2'000 m - ");
  Serial.println(simpledistance( 50.123,  7.123,  50.14, 7.14)); //2 km
}


const char *cardinalDE(double course){
  //German translation: E -> O...
  static const char* directions[] = {"N", "NNO", "NO", "ONO", "O", "OSO", "SO", "SSO", "S", "SSW", "SW", "WSW", "W", "WNW", "NW", "NNW"};
  int direction = (int)((course + 11.25f) / 22.5f);
  return directions[direction % 16];
}
